<?php
/**
 * pesees_secured.php - API sécurisée pour la gestion des pesées avec compartimentalisation par sites
 *
 * Cette version intègre:
 * - Authentification JWT obligatoire
 * - Filtrage automatique par sites selon les droits de l'utilisateur
 * - Utilisation de la configuration centralisée
 */

require_once 'config.php';
require_once 'JwtManager.php';
require_once 'SiteAccessControl.php';

// Headers CORS
Utils::setCorsHeaders();

try {
    $action = $_GET['action'] ?? 'info';

    // Actions publiques (sans authentification)
    $publicActions = ['info', 'version'];

    // Vérifier l'authentification pour les actions protégées
    $userPayload = null;
    $userSites = [];

    if (!in_array($action, $publicActions)) {
        $userPayload = JwtManager::requireAuth();
        $userSites = SiteAccessControl::getUserSites($userPayload['user_id']);
    }

    // Connexion à la base de données
    $pdo = DatabaseConfig::getConnection();

    switch ($action) {
        case 'info':
            // Informations sur l'API
            Utils::jsonResponse([
                'api_name' => 'Pesées API Sécurisée',
                'version' => '2.0.0',
                'features' => [
                    'jwt_authentication',
                    'site_compartmentalization',
                    'role_based_access'
                ],
                'available_actions' => [
                    'test', 'stats', 'list', 'search',
                    'ticket', 'charts', 'options', 'version'
                ],
                'authentication' => 'JWT Bearer Token required (except info and version)'
            ], true, 'API Pesées Sécurisée');
            break;

        case 'test':
            // Test avec filtrage par sites
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT COUNT(*) as total FROM pesee WHERE 1=1",
                $userSites
            );

            $stmt = $pdo->prepare($filter['sql']);
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $count = $stmt->fetch()['total'];

            Utils::jsonResponse([
                'database_test' => 'OK',
                'total_pesees' => $count,
                'user_id' => $userPayload['user_id'],
                'user_name' => $userPayload['username'],
                'accessible_sites' => count($userSites),
                'has_site_restriction' => !empty($userSites),
                'test_time' => date('Y-m-d H:i:s')
            ], true, 'Test avec authentification réussi');
            break;

        case 'stats':
            // Statistiques filtrées par sites
            $stats = [];

            // Total pesées accessibles
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT COUNT(*) as total FROM pesee WHERE 1=1",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql']);
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $stats['totalPesees'] = $stmt->fetch()['total'];

            // Poids total
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT SUM(PoidsNet) as total FROM pesee WHERE PoidsNet > 0",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql']);
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $totalPoids = $stmt->fetch()['total'] ?? 0;
            $stats['poidsTotal'] = round($totalPoids / 1000, 2);

            // Pesées aujourd'hui
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT COUNT(*) as total FROM pesee WHERE DATE(dmv) = CURDATE()",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql']);
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $stats['peseesAujourdhui'] = $stmt->fetch()['total'];

            // Clients actifs
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT COUNT(DISTINCT NomClient) as total FROM pesee WHERE NomClient IS NOT NULL AND NomClient != ''",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql']);
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $stats['clientsActifs'] = $stmt->fetch()['total'];

            // Sites accessibles
            $stats['sitesAccessibles'] = empty($userSites) ? 'Tous' : count($userSites);

            Utils::jsonResponse($stats, true, 'Statistiques récupérées');
            break;

        case 'list':
            // Liste des pesées avec pagination et filtrage par sites
            $limit = max(1, min(100, (int)($_GET['limit'] ?? 50)));
            $page = max(1, (int)($_GET['page'] ?? 1));
            $offset = ($page - 1) * $limit;

            // Appliquer le filtre de sites
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT ID, CodePesee, NumTicket, dmv, NomClient, NomFournisseur, NomProduit,
                        Poids1, Poids2, PoidsNet, Immatriculation, NomPeseur1, CodeSite, NomSite, Mouvement
                 FROM pesee WHERE 1=1",
                $userSites
            );

            $sql = $filter['sql'] . " ORDER BY ID DESC LIMIT ? OFFSET ?";
            $stmt = $pdo->prepare($sql);

            // Bind des paramètres de sites
            $paramIndex = 1;
            foreach ($filter['params'] as $param) {
                $stmt->bindValue($paramIndex++, $param);
            }

            // Bind des paramètres de pagination
            $stmt->bindValue($paramIndex++, $limit, PDO::PARAM_INT);
            $stmt->bindValue($paramIndex, $offset, PDO::PARAM_INT);

            $stmt->execute();
            $pesees = $stmt->fetchAll();

            Utils::jsonResponse([
                'pesees' => $pesees,
                'count' => count($pesees),
                'page' => $page,
                'limit' => $limit,
                'has_more' => count($pesees) === $limit,
                'site_filter_applied' => !empty($userSites)
            ], true, count($pesees) . ' pesées récupérées (page ' . $page . ')');
            break;

        case 'search':
            // Recherche avancée avec filtrage par sites
            $conditions = [];
            $params = [];

            // Critères de recherche
            if (!empty($_GET['codePesee'])) {
                $conditions[] = "CodePesee LIKE ?";
                $params[] = '%' . $_GET['codePesee'] . '%';
            }

            if (!empty($_GET['numTicket'])) {
                $conditions[] = "NumTicket LIKE ?";
                $params[] = '%' . $_GET['numTicket'] . '%';
            }

            if (!empty($_GET['client'])) {
                $conditions[] = "NomClient LIKE ?";
                $params[] = '%' . $_GET['client'] . '%';
            }

            if (!empty($_GET['fournisseur'])) {
                $conditions[] = "NomFournisseur LIKE ?";
                $params[] = '%' . $_GET['fournisseur'] . '%';
            }

            if (!empty($_GET['mouvement'])) {
                $conditions[] = "Mouvement = ?";
                $params[] = $_GET['mouvement'];
            }

            if (!empty($_GET['produit'])) {
                $conditions[] = "NomProduit LIKE ?";
                $params[] = '%' . $_GET['produit'] . '%';
            }

            if (!empty($_GET['dateDebut'])) {
                $conditions[] = "dmv >= ?";
                $params[] = $_GET['dateDebut'];
            }

            if (!empty($_GET['dateFin'])) {
                $conditions[] = "dmv <= ?";
                $params[] = $_GET['dateFin'];
            }

            if (!empty($_GET['immatriculation'])) {
                $conditions[] = "Immatriculation LIKE ?";
                $params[] = '%' . $_GET['immatriculation'] . '%';
            }

            // Construire la requête de base
            $whereClause = empty($conditions) ? 'WHERE 1=1' : 'WHERE ' . implode(' AND ', $conditions);

            $baseQuery = "SELECT
                ID, CodePesee, NumTicket, dmv, NomClient, NomFournisseur, NomProduit,
                Poids1, Poids2, PoidsNet, Immatriculation, NomPeseur1, Provenance, Destination,
                CodeSite, NomSite, Mouvement
                FROM pesee
                $whereClause";

            // Appliquer le filtre de sites
            $filter = SiteAccessControl::applySiteFilter($baseQuery, $userSites);
            $sql = $filter['sql'] . " ORDER BY ID DESC LIMIT 100";

            // Combiner tous les paramètres
            $allParams = array_merge($params, $filter['params']);

            $stmt = $pdo->prepare($sql);
            foreach ($allParams as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }

            $stmt->execute();
            $results = $stmt->fetchAll();

            // Calculer le poids total
            $totalWeight = array_sum(array_column($results, 'PoidsNet'));
            $totalWeightTonnes = round($totalWeight / 1000, 2);

            Utils::jsonResponse([
                'results' => $results,
                'count' => count($results),
                'totalWeight' => $totalWeight,
                'totalWeightTonnes' => $totalWeightTonnes,
                'site_filter_applied' => !empty($userSites)
            ], true, count($results) . ' résultats trouvés (Poids total: ' . $totalWeightTonnes . ' tonnes)');
            break;

        case 'ticket':
            // Récupérer un ticket spécifique avec vérification d'accès au site
            $numTicket = $_GET['numTicket'] ?? '';
            $codePesee = $_GET['codePesee'] ?? '';

            if (empty($numTicket) && empty($codePesee)) {
                Utils::errorResponse('Numéro de ticket ou code pesée requis');
            }

            $searchValue = !empty($numTicket) ? $numTicket : $codePesee;

            // Rechercher le ticket avec filtrage par sites
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT * FROM pesee WHERE (NumTicket = ? OR CodePesee = ?)",
                $userSites
            );

            $stmt = $pdo->prepare($filter['sql'] . " LIMIT 1");
            $stmt->bindValue(1, $searchValue);
            $stmt->bindValue(2, $searchValue);

            // Bind des paramètres de sites
            $paramIndex = 3;
            foreach ($filter['params'] as $param) {
                $stmt->bindValue($paramIndex++, $param);
            }

            $stmt->execute();
            $ticket = $stmt->fetch();

            if ($ticket) {
                Utils::jsonResponse($ticket, true, 'Ticket trouvé');
            } else {
                Utils::jsonResponse(null, false, "Ticket non trouvé ou non accessible pour: $searchValue");
            }
            break;

        case 'options':
            // Options filtrées par sites accessibles
            $options = [];

            $filter = SiteAccessControl::applySiteFilter(
                "SELECT DISTINCT NomClient FROM pesee WHERE NomClient IS NOT NULL AND NomClient != ''",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql'] . " ORDER BY NomClient");
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $options['clients'] = array_column($stmt->fetchAll(), 'NomClient');

            $filter = SiteAccessControl::applySiteFilter(
                "SELECT DISTINCT NomFournisseur FROM pesee WHERE NomFournisseur IS NOT NULL AND NomFournisseur != ''",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql'] . " ORDER BY NomFournisseur");
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $options['fournisseurs'] = array_column($stmt->fetchAll(), 'NomFournisseur');

            // Sites accessibles
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT DISTINCT CodeSite, NomSite FROM pesee WHERE CodeSite IS NOT NULL AND CodeSite != ''",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql'] . " ORDER BY NomSite");
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $sites = $stmt->fetchAll();
            $options['sites'] = $sites;
            $options['codesSites'] = array_column($sites, 'CodeSite');

            // Autres options...
            $options['mouvements'] = ['RECEPTION', 'EXPEDITION', 'TRANSFERT', 'RETOUR', 'CONTROLE'];

            Utils::jsonResponse($options, true, 'Options récupérées (filtrées par sites)');
            break;

        case 'charts':
            // Données pour graphiques filtrées par sites
            $chartData = [];

            // Évolution mensuelle
            $filter = SiteAccessControl::applySiteFilter(
                "SELECT
                    DATE_FORMAT(dmv, '%Y-%m') as mois,
                    COUNT(*) as nombre,
                    SUM(PoidsNet) as poids
                FROM pesee
                WHERE dmv >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)",
                $userSites
            );
            $stmt = $pdo->prepare($filter['sql'] . " GROUP BY DATE_FORMAT(dmv, '%Y-%m') ORDER BY mois");
            foreach ($filter['params'] as $i => $param) {
                $stmt->bindValue($i + 1, $param);
            }
            $stmt->execute();
            $chartData['evolutionMensuelle'] = $stmt->fetchAll();

            Utils::jsonResponse($chartData, true, 'Données graphiques récupérées');
            break;

        case 'version':
            // Version de l'API
            Utils::jsonResponse([
                'api_version' => '2.0.0',
                'app_version' => '1.5.2',
                'features' => [
                    'site_compartmentalization' => true,
                    'jwt_authentication' => true,
                    'secured_access' => true
                ],
                'last_updated' => '2025-10-08'
            ], true, 'Version de l\'API');
            break;

        default:
            Utils::errorResponse('Action non reconnue: ' . $action);
    }

} catch (Exception $e) {
    Utils::logError("Erreur dans pesees_secured.php", [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    Utils::errorResponse('Erreur: ' . $e->getMessage(), 500);
}
?>
