<?php
// pesees.php - API améliorée pour la gestion des pesées
// require_once 'config.php'; // Temporairement désactivé pour debug

// Activer l'affichage des erreurs pour debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Headers CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Gérer les requêtes OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Fonction pour retourner du JSON
function jsonResponse($data, $success = true, $message = '') {
    $response = [
        'success' => $success,
        'data' => $data,
        'message' => $message,
        'timestamp' => date('Y-m-d H:i:s'),
        'debug_info' => [
            'php_version' => phpversion(),
            'request_method' => $_SERVER['REQUEST_METHOD'],
            'request_uri' => $_SERVER['REQUEST_URI'],
            'get_params' => $_GET
        ]
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}

// Fonction pour retourner une erreur
function errorResponse($message, $httpCode = 400) {
    http_response_code($httpCode);
    jsonResponse(null, false, $message);
}

try {
    // Test de base - vérifier que le script s'exécute
    if (!isset($_GET['action'])) {
        jsonResponse([
            'available_actions' => ['test', 'stats', 'list', 'search', 'ticket', 'charts', 'options', 'version'],
            'usage' => 'Ajoutez ?action=test pour tester'
        ], true, 'API Pesées Améliorée - Test OK');
    }
    
    // Debug spécial pour l'action ticket
    if ($_GET['action'] === 'ticket') {
        error_log("DEBUG: Début action ticket - " . date('Y-m-d H:i:s'));
        error_log("DEBUG: GET parameters: " . json_encode($_GET));
    }
    
    $action = $_GET['action'];
    
    // Connexion directe pour éviter EnvLoader
    $dbHost = 'serverkmaf.helioho.st';
    $dbName = 'kombarferd_gestpeseedb';
    $dbUser = 'kombarferd_dbadmin';
    $dbPass = 'dbadmin@2024';
    
    $dsn = "mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4";
    $pdo = new PDO($dsn, $dbUser, $dbPass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    
    switch ($action) {
        case 'test':
            // Test simple
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM pesee");
            $count = $stmt->fetch()['total'];
            
            jsonResponse([
                'database_test' => 'OK',
                'total_pesees' => $count,
                'test_time' => date('Y-m-d H:i:s')
            ], true, 'Test de base réussi');
            break;
            
        case 'stats':
            // Statistiques simples
            $stats = [];
            
            // Total pesées
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM pesee");
            $stats['totalPesees'] = $stmt->fetch()['total'];
            
            // Poids total
            $stmt = $pdo->query("SELECT SUM(PoidsNet) as total FROM pesee WHERE PoidsNet > 0");
            $totalPoids = $stmt->fetch()['total'] ?? 0;
            $stats['poidsTotal'] = round($totalPoids / 1000, 2); // En tonnes
            
            // Pesées aujourd'hui
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM pesee WHERE DATE(dmv) = CURDATE()");
            $stats['peseesAujourdhui'] = $stmt->fetch()['total'];
            
            // Clients actifs
            $stmt = $pdo->query("SELECT COUNT(DISTINCT NomClient) as total FROM pesee WHERE NomClient IS NOT NULL AND NomClient != ''");
            $stats['clientsActifs'] = $stmt->fetch()['total'];
            
            jsonResponse($stats, true, 'Statistiques récupérées');
            break;
            
        case 'list':
            // Validation simple des paramètres
            $limit = max(1, min(100, (int)($_GET['limit'] ?? 50)));
            $page = max(1, (int)($_GET['page'] ?? 1));
            $offset = ($page - 1) * $limit;
            
            $stmt = $pdo->prepare("SELECT 
                ID, CodePesee, NumTicket, dmv, NomClient, NomFournisseur, NomProduit, 
                Poids1, Poids2, PoidsNet, Immatriculation, NomPeseur1, CodeSite, NomSite, Mouvement
                FROM pesee 
                ORDER BY ID DESC 
                LIMIT :limit OFFSET :offset");
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $pesees = $stmt->fetchAll();
            
            jsonResponse([
                'pesees' => $pesees,
                'count' => count($pesees),
                'page' => $page,
                'limit' => $limit,
                'has_more' => count($pesees) === $limit
            ], true, count($pesees) . ' pesées récupérées (page ' . $page . ')');
            break;
            
        case 'search':
            // Recherche avancée avec tous les critères
            $conditions = [];
            $params = [];
            
            // Construire la requête dynamiquement selon les critères fournis
            if (!empty($_GET['codePesee'])) {
                $conditions[] = "CodePesee LIKE :codePesee";
                $params[':codePesee'] = '%' . $_GET['codePesee'] . '%';
            }
            
            if (!empty($_GET['numTicket'])) {
                $conditions[] = "NumTicket LIKE :numTicket";
                $params[':numTicket'] = '%' . $_GET['numTicket'] . '%';
            }
            
            if (!empty($_GET['client'])) {
                $conditions[] = "NomClient LIKE :client";
                $params[':client'] = '%' . $_GET['client'] . '%';
            }
            
            if (!empty($_GET['fournisseur'])) {
                $conditions[] = "NomFournisseur LIKE :fournisseur";
                $params[':fournisseur'] = '%' . $_GET['fournisseur'] . '%';
            }
            
            if (!empty($_GET['mouvement'])) {
                $conditions[] = "Mouvement = :mouvement";
                $params[':mouvement'] = $_GET['mouvement'];
                
                // Debug: Log le critère de mouvement
                error_log("DEBUG: Filtrage par mouvement = '" . $_GET['mouvement'] . "'");
            }
            
            if (!empty($_GET['codeSite'])) {
                $conditions[] = "CodeSite LIKE :codeSite";
                $params[':codeSite'] = '%' . $_GET['codeSite'] . '%';
            }
            
            if (!empty($_GET['site'])) {
                $conditions[] = "NomSite LIKE :site";
                $params[':site'] = '%' . $_GET['site'] . '%';
            }
            
            if (!empty($_GET['produit'])) {
                $conditions[] = "NomProduit LIKE :produit";
                $params[':produit'] = '%' . $_GET['produit'] . '%';
            }
            
            if (!empty($_GET['provenance'])) {
                $conditions[] = "Provenance LIKE :provenance";
                $params[':provenance'] = '%' . $_GET['provenance'] . '%';
            }
            
            if (!empty($_GET['destination'])) {
                $conditions[] = "Destination LIKE :destination";
                $params[':destination'] = '%' . $_GET['destination'] . '%';
            }
            
            if (!empty($_GET['dateDebut'])) {
                $conditions[] = "dmv >= :dateDebut";
                $params[':dateDebut'] = $_GET['dateDebut'];
            }
            
            if (!empty($_GET['dateFin'])) {
                $conditions[] = "dmv <= :dateFin";
                $params[':dateFin'] = $_GET['dateFin'];
            }
            
            if (!empty($_GET['immatriculation'])) {
                $conditions[] = "Immatriculation LIKE :immatriculation";
                $params[':immatriculation'] = '%' . $_GET['immatriculation'] . '%';
            }
            
            // Si aucun critère, retourner toutes les pesées (limitées)
            $whereClause = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);
            
            $sql = "SELECT 
                ID, CodePesee, NumTicket, dmv, NomClient, NomFournisseur, NomProduit, 
                Poids1, Poids2, PoidsNet, Immatriculation, NomPeseur1, Provenance, Destination,
                CodeSite, NomSite, Mouvement
                FROM pesee 
                $whereClause
                ORDER BY ID DESC 
                LIMIT 100";
            
            // Debug: Log la requête SQL
            error_log("DEBUG: SQL = " . $sql);
            error_log("DEBUG: Paramètres = " . json_encode($params));
            
            $stmt = $pdo->prepare($sql);
            
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value, PDO::PARAM_STR);
            }
            
            $stmt->execute();
            $results = $stmt->fetchAll();
            
            // Debug: Log les résultats
            error_log("DEBUG: Nombre de résultats = " . count($results));
            if (!empty($results)) {
                error_log("DEBUG: Premier résultat = " . json_encode($results[0]));
                // Vérifier spécifiquement le champ Mouvement
                if (isset($results[0]['Mouvement'])) {
                    error_log("DEBUG: Champ Mouvement du premier résultat = '" . $results[0]['Mouvement'] . "'");
                } else {
                    error_log("DEBUG: ⚠️ ATTENTION: Champ Mouvement absent du premier résultat !");
                    error_log("DEBUG: Clés disponibles = " . json_encode(array_keys($results[0])));
                }
            }
            
            // Calculer le poids total des résultats
            $totalWeight = 0;
            foreach ($results as $result) {
                $totalWeight += floatval($result['PoidsNet'] ?? 0);
            }
            
            // Convertir en tonnes
            $totalWeightTonnes = round($totalWeight / 1000, 2);
            
            // Préparer la réponse avec les métadonnées
            $response = [
                'results' => $results,
                'count' => count($results),
                'totalWeight' => $totalWeight,
                'totalWeightTonnes' => $totalWeightTonnes
            ];
            
            jsonResponse($response, true, count($results) . ' résultats trouvés (Poids total: ' . $totalWeightTonnes . ' tonnes)');
            break;
            
        case 'ticket':
            // Récupérer un ticket spécifique
            $numTicket = $_GET['numTicket'] ?? '';
            $codePesee = $_GET['codePesee'] ?? '';
            
            if (empty($numTicket) && empty($codePesee)) {
                jsonResponse(null, false, 'Numéro de ticket ou code pesée requis');
            }
            
            $searchValue = !empty($numTicket) ? $numTicket : $codePesee;
            $ticket = null;
            
            // Essayer d'abord avec NumTicket
            $sql = "SELECT * FROM pesee WHERE NumTicket = :param";
            $stmt = $pdo->prepare($sql);
            $stmt->bindParam(':param', $searchValue, PDO::PARAM_STR);
            $stmt->execute();
            $ticket = $stmt->fetch();
            
            // Si pas trouvé avec NumTicket, essayer avec CodePesee
            if (!$ticket) {
                $sql = "SELECT * FROM pesee WHERE CodePesee = :param";
                $stmt = $pdo->prepare($sql);
                $stmt->bindParam(':param', $searchValue, PDO::PARAM_STR);
                $stmt->execute();
                $ticket = $stmt->fetch();
            }
            
            // Si toujours pas trouvé, essayer une recherche approximative (LIKE)
            if (!$ticket) {
                $sql = "SELECT * FROM pesee WHERE NumTicket LIKE :param OR CodePesee LIKE :param ORDER BY ID DESC LIMIT 1";
                $likeParam = '%' . $searchValue . '%';
                $stmt = $pdo->prepare($sql);
                $stmt->bindParam(':param', $likeParam, PDO::PARAM_STR);
                $stmt->execute();
                $ticket = $stmt->fetch();
            }
            
            if ($ticket) {
                jsonResponse($ticket, true, 'Ticket trouvé');
            } else {
                jsonResponse(null, false, "Ticket non trouvé pour: $searchValue");
            }
            break;
            
        case 'test_structure':
            // Test spécial pour vérifier la structure de la table
            try {
                // Vérifier si la colonne Mouvement existe
                $stmt = $pdo->query("DESCRIBE pesee");
                $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $columnNames = array_column($columns, 'Field');
                $mouvementExists = in_array('Mouvement', $columnNames);
                
                error_log("DEBUG STRUCTURE: Colonnes de la table pesee = " . json_encode($columnNames));
                error_log("DEBUG STRUCTURE: Colonne Mouvement existe = " . ($mouvementExists ? 'OUI' : 'NON'));
                
                $result = [
                    'table_columns' => $columnNames,
                    'mouvement_column_exists' => $mouvementExists,
                    'total_columns' => count($columnNames)
                ];
                
                // Si la colonne existe, tester son contenu
                if ($mouvementExists) {
                    try {
                        $stmt = $pdo->query("SELECT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' LIMIT 5");
                        $exemples = $stmt->fetchAll(PDO::FETCH_COLUMN);
                        $result['exemples_mouvements'] = $exemples;
                        
                        $stmt = $pdo->query("SELECT DISTINCT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' ORDER BY Mouvement");
                        $distincts = $stmt->fetchAll(PDO::FETCH_COLUMN);
                        $result['mouvements_distincts'] = $distincts;
                        
                        error_log("DEBUG STRUCTURE: Mouvements distincts = " . json_encode($distincts));
                    } catch (Exception $e) {
                        $result['erreur_contenu'] = $e->getMessage();
                    }
                }
                
                jsonResponse($result, true, 'Test structure table');
                
            } catch (PDOException $e) {
                error_log("DEBUG STRUCTURE: Erreur = " . $e->getMessage());
                jsonResponse(['erreur' => $e->getMessage()], false, 'Erreur test structure');
            }
            break;
            
        case 'test_mouvements':
            // Test spécial pour voir les mouvements en base
            try {
                $stmt = $pdo->query("SELECT DISTINCT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' ORDER BY Mouvement");
                $mouvements = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                error_log("DEBUG TEST: Mouvements bruts de la base = " . json_encode($mouvements));
                
                jsonResponse([
                    'mouvements_bruts' => $mouvements,
                    'count' => count($mouvements),
                    'query' => "SELECT DISTINCT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' ORDER BY Mouvement"
                ], true, 'Test mouvements');
            } catch (PDOException $e) {
                error_log("DEBUG TEST: Erreur SQL = " . $e->getMessage());
                jsonResponse(['erreur' => $e->getMessage()], false, 'Erreur colonne Mouvement');
            }
            break;
            
        case 'add_mouvement_column':
            // Ajouter la colonne Mouvement si elle n'existe pas
            try {
                // Vérifier d'abord si elle existe déjà
                $stmt = $pdo->query("DESCRIBE pesee");
                $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $columnNames = array_column($columns, 'Field');
                
                if (in_array('Mouvement', $columnNames)) {
                    jsonResponse(['message' => 'La colonne Mouvement existe déjà'], true, 'Colonne déjà présente');
                } else {
                    // Ajouter la colonne
                    $pdo->exec("ALTER TABLE pesee ADD COLUMN Mouvement VARCHAR(50) DEFAULT NULL");
                    
                    // Remplir avec des valeurs par défaut pour les lignes existantes
                    $pdo->exec("UPDATE pesee SET Mouvement = 'RECEPTION' WHERE Mouvement IS NULL");
                    
                    error_log("DEBUG: Colonne Mouvement ajoutée avec succès");
                    jsonResponse(['message' => 'Colonne Mouvement ajoutée et remplie avec des valeurs par défaut'], true, 'Colonne ajoutée');
                }
            } catch (PDOException $e) {
                error_log("DEBUG: Erreur ajout colonne: " . $e->getMessage());
                jsonResponse(['erreur' => $e->getMessage()], false, 'Erreur ajout colonne');
            }
            break;
            
        case 'options':
            // Récupérer les listes de valeurs pour pré-remplir les formulaires
            $options = [];
            
            // Liste des clients
            $stmt = $pdo->query("SELECT DISTINCT NomClient FROM pesee WHERE NomClient IS NOT NULL AND NomClient != '' ORDER BY NomClient");
            $options['clients'] = array_column($stmt->fetchAll(), 'NomClient');
            
            // Liste des fournisseurs
            $stmt = $pdo->query("SELECT DISTINCT NomFournisseur FROM pesee WHERE NomFournisseur IS NOT NULL AND NomFournisseur != '' ORDER BY NomFournisseur");
            $options['fournisseurs'] = array_column($stmt->fetchAll(), 'NomFournisseur');
            
            // Liste des mouvements - debugging renforcé
            try {
                error_log("DEBUG: Tentative de récupération des mouvements...");
                
                // D'abord, compter combien de lignes ont un mouvement
                $countStmt = $pdo->query("SELECT COUNT(*) as total FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != ''");
                $count = $countStmt->fetch()['total'];
                error_log("DEBUG: Nombre de lignes avec mouvement non vide = " . $count);
                
                // Puis récupérer tous les mouvements avec quelques exemples
                $stmt = $pdo->query("SELECT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' LIMIT 10");
                $exemples = $stmt->fetchAll(PDO::FETCH_COLUMN);
                error_log("DEBUG: Exemples de mouvements = " . json_encode($exemples));
                
                // Enfin, les mouvements distincts
                $stmt = $pdo->query("SELECT DISTINCT Mouvement FROM pesee WHERE Mouvement IS NOT NULL AND Mouvement != '' ORDER BY Mouvement");
                $mouvements = array_column($stmt->fetchAll(), 'Mouvement');
                
                error_log("DEBUG: Mouvements distincts trouvés = " . json_encode($mouvements));
                error_log("DEBUG: Nombre de mouvements distincts = " . count($mouvements));
                
                // Si aucun mouvement trouvé, utiliser des valeurs par défaut
                if (empty($mouvements)) {
                    $mouvements = ['RECEPTION', 'EXPEDITION', 'TRANSFERT', 'RETOUR', 'CONTROLE'];
                    error_log("DEBUG: Utilisation des mouvements par défaut car aucun trouvé");
                }
                $options['mouvements'] = $mouvements;
            } catch (PDOException $e) {
                // Si le champ n'existe pas, utiliser des valeurs par défaut
                error_log("DEBUG: ERREUR SQL champ Mouvement: " . $e->getMessage());
                $options['mouvements'] = ['RECEPTION', 'EXPEDITION', 'TRANSFERT', 'RETOUR', 'CONTROLE'];
            }
            
            // Liste des produits
            $stmt = $pdo->query("SELECT DISTINCT NomProduit FROM pesee WHERE NomProduit IS NOT NULL AND NomProduit != '' ORDER BY NomProduit");
            $options['produits'] = array_column($stmt->fetchAll(), 'NomProduit');
            
            // Liste des codes de sites
            $stmt = $pdo->query("SELECT DISTINCT CodeSite FROM pesee WHERE CodeSite IS NOT NULL AND CodeSite != '' ORDER BY CodeSite");
            $options['codesSites'] = array_column($stmt->fetchAll(), 'CodeSite');
            
            // Liste des noms de sites
            $stmt = $pdo->query("SELECT DISTINCT NomSite FROM pesee WHERE NomSite IS NOT NULL AND NomSite != '' ORDER BY NomSite");
            $options['sites'] = array_column($stmt->fetchAll(), 'NomSite');
            
            // Liste des provenances
            $stmt = $pdo->query("SELECT DISTINCT Provenance FROM pesee WHERE Provenance IS NOT NULL AND Provenance != '' ORDER BY Provenance");
            $options['provenances'] = array_column($stmt->fetchAll(), 'Provenance');
            
            // Liste des destinations
            $stmt = $pdo->query("SELECT DISTINCT Destination FROM pesee WHERE Destination IS NOT NULL AND Destination != '' ORDER BY Destination");
            $options['destinations'] = array_column($stmt->fetchAll(), 'Destination');
            
            // Liste des peseurs
            $stmt = $pdo->query("SELECT DISTINCT NomPeseur1 FROM pesee WHERE NomPeseur1 IS NOT NULL AND NomPeseur1 != '' ORDER BY NomPeseur1");
            $options['peseurs'] = array_column($stmt->fetchAll(), 'NomPeseur1');
            
            jsonResponse($options, true, 'Options récupérées');
            break;
            
        case 'charts':
            // Données pour les graphiques
            $chartData = [];
            
            // Évolution mensuelle
            $stmt = $pdo->query("
                SELECT 
                    DATE_FORMAT(dmv, '%Y-%m') as mois,
                    COUNT(*) as nombre,
                    SUM(PoidsNet) as poids
                FROM pesee 
                WHERE dmv >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
                GROUP BY DATE_FORMAT(dmv, '%Y-%m')
                ORDER BY mois
            ");
            $chartData['evolutionMensuelle'] = $stmt->fetchAll();
            
            // Répartition par produit
            $stmt = $pdo->query("
                SELECT 
                    NomProduit as produit,
                    COUNT(*) as nombre
                FROM pesee 
                WHERE NomProduit IS NOT NULL AND NomProduit != ''
                GROUP BY NomProduit
                ORDER BY nombre DESC
                LIMIT 10
            ");
            $chartData['repartitionProduits'] = $stmt->fetchAll();
            
            // Top clients
            $stmt = $pdo->query("
                SELECT 
                    NomClient as client,
                    COUNT(*) as nombre,
                    SUM(PoidsNet) as poids_total
                FROM pesee 
                WHERE NomClient IS NOT NULL AND NomClient != ''
                GROUP BY NomClient
                ORDER BY nombre DESC
                LIMIT 10
            ");
            $chartData['topClients'] = $stmt->fetchAll();
            
            // Évolution hebdomadaire (4 dernières semaines)
            $stmt = $pdo->query("
                SELECT 
                    WEEK(dmv) as semaine,
                    COUNT(*) as nombre,
                    SUM(PoidsNet) as poids
                FROM pesee 
                WHERE dmv >= DATE_SUB(CURDATE(), INTERVAL 4 WEEK)
                GROUP BY WEEK(dmv)
                ORDER BY semaine
            ");
            $chartData['evolutionHebdomadaire'] = $stmt->fetchAll();
            
            jsonResponse($chartData, true, 'Données graphiques récupérées');
            break;
            
        case 'version':
            // Informations de version de l'application
            $manifestPath = __DIR__ . '/manifest.json';
            $version = [
                'app_version' => '1.0.0',
                'api_version' => '1.0.0',
                'last_updated' => date('Y-m-d H:i:s'),
                'build_number' => date('YmdHis'),
                'cache_version' => 'kombar-ferd-v1.0.0'
            ];
            
            // Lire la version du manifest si disponible
            if (file_exists($manifestPath)) {
                $manifestContent = file_get_contents($manifestPath);
                $manifest = json_decode($manifestContent, true);
                if ($manifest && isset($manifest['version'])) {
                    $version['app_version'] = $manifest['version'];
                }
                $version['manifest_last_modified'] = date('Y-m-d H:i:s', filemtime($manifestPath));
            }
            
            // Ajouter les timestamps des fichiers principaux
            $mainFiles = [
                'pesee_webapp.html' => __DIR__ . '/pesee_webapp.html',
                'sw.js' => __DIR__ . '/sw.js',
                'notification-watcher.js' => __DIR__ . '/notification-watcher.js'
            ];
            
            foreach ($mainFiles as $name => $path) {
                if (file_exists($path)) {
                    $version['files'][$name] = [
                        'last_modified' => date('Y-m-d H:i:s', filemtime($path)),
                        'size' => filesize($path)
                    ];
                }
            }
            
            jsonResponse($version, true, 'Informations de version récupérées');
            break;
            
        default:
            errorResponse('Action non reconnue: ' . $action);
    }
    
} catch (PDOException $e) {
    errorResponse('Erreur de base de données: ' . $e->getMessage(), 500);
} catch (Exception $e) {
    errorResponse('Erreur: ' . $e->getMessage(), 500);
}
?>