// charts-fournisseur.js - Graphiques spécifiques aux fournisseurs

/**
 * Créer le graphique Top 10 Fournisseurs
 */
function createTopFournisseursChart(data) {
    const ctx = document.getElementById('topFournisseursChart');
    if (!ctx || !data || data.length === 0) {
        console.warn('❌ Canvas topFournisseursChart non trouvé ou pas de données');
        return;
    }
    
    // Grouper par fournisseur avec poids total
    const fournisseurs = {};
    data.forEach(item => {
        const fournisseur = item.NomFournisseur || 'Non spécifié';
        if (!fournisseurs[fournisseur]) {
            fournisseurs[fournisseur] = { count: 0, weight: 0 };
        }
        fournisseurs[fournisseur].count++;
        fournisseurs[fournisseur].weight += parseFloat(item.PoidsNet) || 0;
    });
    
    // Trier par poids et prendre les top 10
    const sortedFournisseurs = Object.entries(fournisseurs)
        .sort(([,a], [,b]) => b.weight - a.weight)
        .slice(0, 10);
    
    // Détruire le graphique existant s'il y en a un
    if (window.reportsData && window.reportsData.charts && window.reportsData.charts.topFournisseurs) {
        console.log('🗑️ Destruction ancien graphique topFournisseurs');
        try {
            window.reportsData.charts.topFournisseurs.destroy();
            window.reportsData.charts.topFournisseurs = null;
        } catch (e) {
            console.warn('⚠️ Erreur destruction graphique:', e);
        }
    }

    // Solution robuste : chercher et détruire tout graphique existant sur ce canvas
    const chartInstances = Chart.getChart(ctx);
    if (chartInstances) {
        console.log('🗑️ Destruction instance Chart trouvée sur canvas');
        chartInstances.destroy();
    }
    
    try {
        const chart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: sortedFournisseurs.map(([name]) => 
                    name.length > 15 ? name.substring(0, 15) + '...' : name
                ),
                datasets: [{
                    label: 'Poids total (kg)',
                    data: sortedFournisseurs.map(([, data]) => data.weight),
                    backgroundColor: 'rgba(118, 75, 162, 0.8)',
                    borderColor: 'rgb(118, 75, 162)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        labels: { color: '#f1f5f9' }
                    },
                    tooltip: {
                        backgroundColor: 'rgba(15, 23, 42, 0.9)',
                        titleColor: '#f1f5f9',
                        bodyColor: '#f1f5f9',
                        callbacks: {
                            label: function(context) {
                                const fournisseurData = sortedFournisseurs[context.dataIndex];
                                return [
                                    `Poids total: ${context.parsed.y.toLocaleString()} kg`,
                                    `Nombre de pesées: ${fournisseurData[1].count}`
                                ];
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        ticks: { color: '#94a3b8' },
                        grid: { color: 'rgba(148, 163, 184, 0.1)' }
                    },
                    y: {
                        ticks: { 
                            color: '#94a3b8',
                            callback: function(value) {
                                return value.toLocaleString() + ' kg';
                            }
                        },
                        grid: { color: 'rgba(148, 163, 184, 0.1)' }
                    }
                }
            }
        });
        
        // Stocker la référence du graphique
        if (window.reportsData && window.reportsData.charts) {
            window.reportsData.charts.topFournisseurs = chart;
        }
        
        console.log('✅ Graphique Top Fournisseurs créé');
    } catch (error) {
        console.error('❌ Erreur création graphique fournisseurs:', error);
    }
}

/**
 * Créer le graphique de répartition Client-Fournisseur
 */
function createClientFournisseurChart(data) {
    const ctx = document.getElementById('clientFournisseurChart');
    if (!ctx || !data || data.length === 0) {
        console.warn('❌ Canvas clientFournisseurChart non trouvé ou pas de données');
        return;
    }
    
    // Analyser les relations client-fournisseur
    const relations = {};
    data.forEach(item => {
        const client = item.NomClient || 'Client non spécifié';
        const fournisseur = item.NomFournisseur || 'Fournisseur non spécifié';
        const key = `${client} → ${fournisseur}`;
        
        if (!relations[key]) {
            relations[key] = { count: 0, weight: 0, client, fournisseur };
        }
        relations[key].count++;
        relations[key].weight += parseFloat(item.PoidsNet) || 0;
    });
    
    // Prendre les top 10 relations par poids
    const sortedRelations = Object.entries(relations)
        .sort(([,a], [,b]) => b.weight - a.weight)
        .slice(0, 10);
    
    // Détruire le graphique existant s'il y en a un
    if (window.reportsData && window.reportsData.charts && window.reportsData.charts.clientFournisseur) {
        console.log('🗑️ Destruction ancien graphique clientFournisseur');
        try {
            window.reportsData.charts.clientFournisseur.destroy();
            window.reportsData.charts.clientFournisseur = null;
        } catch (e) {
            console.warn('⚠️ Erreur destruction graphique clientFournisseur:', e);
        }
    }

    // Solution robuste : chercher et détruire tout graphique existant sur ce canvas
    const chartInstances = Chart.getChart(ctx);
    if (chartInstances) {
        console.log('🗑️ Destruction instance Chart trouvée sur canvas clientFournisseur');
        chartInstances.destroy();
    }
    
    try {
        const chart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: sortedRelations.map(([key]) => 
                    key.length > 25 ? key.substring(0, 25) + '...' : key
                ),
                datasets: [{
                    label: 'Poids (kg)',
                    data: sortedRelations.map(([, data]) => data.weight),
                    backgroundColor: [
                        'rgba(102, 126, 234, 0.8)',
                        'rgba(118, 75, 162, 0.8)',
                        'rgba(240, 147, 251, 0.8)',
                        'rgba(79, 172, 254, 0.8)',
                        'rgba(0, 242, 254, 0.8)',
                        'rgba(16, 185, 129, 0.8)',
                        'rgba(245, 87, 108, 0.8)',
                        'rgba(251, 191, 36, 0.8)',
                        'rgba(139, 92, 246, 0.8)',
                        'rgba(236, 72, 153, 0.8)'
                    ],
                    borderColor: [
                        'rgb(102, 126, 234)',
                        'rgb(118, 75, 162)',
                        'rgb(240, 147, 251)',
                        'rgb(79, 172, 254)',
                        'rgb(0, 242, 254)',
                        'rgb(16, 185, 129)',
                        'rgb(245, 87, 108)',
                        'rgb(251, 191, 36)',
                        'rgb(139, 92, 246)',
                        'rgb(236, 72, 153)'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true,
                        position: 'right',
                        labels: { 
                            color: '#f1f5f9',
                            font: { size: 11 },
                            padding: 15
                        }
                    },
                    tooltip: {
                        backgroundColor: 'rgba(15, 23, 42, 0.9)',
                        titleColor: '#f1f5f9',
                        bodyColor: '#f1f5f9',
                        callbacks: {
                            label: function(context) {
                                const relation = sortedRelations[context.dataIndex];
                                const percentage = ((context.parsed / context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0)) * 100).toFixed(1);
                                return [
                                    `${relation[0]}`,
                                    `Poids: ${context.parsed.toLocaleString()} kg (${percentage}%)`,
                                    `Pesées: ${relation[1].count}`
                                ];
                            }
                        }
                    }
                }
            }
        });
        
        // Stocker la référence du graphique
        if (window.reportsData && window.reportsData.charts) {
            window.reportsData.charts.clientFournisseur = chart;
        }
        
        console.log('✅ Graphique Client-Fournisseur créé');
    } catch (error) {
        console.error('❌ Erreur création graphique client-fournisseur:', error);
    }
}

// Export pour utilisation globale
if (typeof window !== 'undefined') {
    window.createTopFournisseursChart = createTopFournisseursChart;
    window.createClientFournisseurChart = createClientFournisseurChart;
}

// Export pour modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        createTopFournisseursChart,
        createClientFournisseurChart
    };
}