<?php
/**
 * SiteAccessControl - Gestion de la compartimentalisation des données par sites
 *
 * Cette classe gère les autorisations d'accès des utilisateurs aux différents sites.
 * Chaque utilisateur peut avoir accès à un ou plusieurs sites spécifiques.
 */

require_once 'config.php';

class SiteAccessControl {

    /**
     * Récupérer les sites accessibles pour un utilisateur
     *
     * @param int $userId ID de l'utilisateur
     * @return array Liste des sites avec code et nom
     */
    public static function getUserSites($userId) {
        try {
            $pdo = DatabaseConfig::getConnection();

            // Récupérer les sites depuis le champ JSON
            $stmt = $pdo->prepare("SELECT sites_acces FROM user WHERE NUt = ? AND actif = 1");
            $stmt->execute([$userId]);
            $result = $stmt->fetch();

            if (!$result) {
                return [];
            }

            // Décoder le JSON
            $sites = json_decode($result['sites_acces'] ?? '[]', true);

            // Si le JSON est invalide ou vide, retourner un tableau vide
            if (!is_array($sites)) {
                Utils::logError("Format JSON invalide pour sites_acces de l'utilisateur $userId");
                return [];
            }

            return $sites;

        } catch (PDOException $e) {
            Utils::logError("Erreur lors de la récupération des sites utilisateur", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Vérifier si un utilisateur a accès à un site spécifique
     *
     * @param int $userId ID de l'utilisateur
     * @param string $codeSite Code du site à vérifier
     * @return bool True si l'utilisateur a accès, False sinon
     */
    public static function canAccessSite($userId, $codeSite) {
        $sites = self::getUserSites($userId);

        // Si aucun site défini, l'utilisateur a accès à tous les sites (ancien comportement)
        if (empty($sites)) {
            return true;
        }

        // Vérifier si le code du site existe dans la liste
        $siteCodes = array_column($sites, 'code');
        return in_array($codeSite, $siteCodes);
    }

    /**
     * Générer une condition WHERE SQL pour filtrer par sites
     *
     * @param array $userSites Tableau des sites de l'utilisateur
     * @param string $siteColumn Nom de la colonne contenant le code site (défaut: CodeSite)
     * @return array ['sql' => condition SQL, 'params' => paramètres à binder]
     */
    public static function getSiteFilterCondition($userSites, $siteColumn = 'CodeSite') {
        // Si l'utilisateur n'a pas de restrictions de sites, ne pas filtrer
        if (empty($userSites)) {
            return [
                'sql' => '1=1',
                'params' => []
            ];
        }

        // Extraire les codes des sites
        $siteCodes = array_column($userSites, 'code');

        if (empty($siteCodes)) {
            return [
                'sql' => '1=1',
                'params' => []
            ];
        }

        // Créer les placeholders pour la requête préparée
        $placeholders = implode(',', array_fill(0, count($siteCodes), '?'));

        return [
            'sql' => "$siteColumn IN ($placeholders)",
            'params' => $siteCodes
        ];
    }

    /**
     * Appliquer un filtre de sites sur une requête SQL
     *
     * @param string $baseQuery Requête SQL de base
     * @param array $userSites Tableau des sites de l'utilisateur
     * @param string $siteColumn Nom de la colonne de site
     * @return array ['sql' => requête complète, 'params' => paramètres]
     */
    public static function applySiteFilter($baseQuery, $userSites, $siteColumn = 'CodeSite') {
        $filter = self::getSiteFilterCondition($userSites, $siteColumn);

        // Déterminer si la requête contient déjà une clause WHERE
        $hasWhere = stripos($baseQuery, 'WHERE') !== false;

        if ($filter['sql'] === '1=1') {
            return [
                'sql' => $baseQuery,
                'params' => []
            ];
        }

        $connector = $hasWhere ? ' AND ' : ' WHERE ';

        return [
            'sql' => $baseQuery . $connector . $filter['sql'],
            'params' => $filter['params']
        ];
    }

    /**
     * Assigner des sites à un utilisateur
     *
     * @param int $userId ID de l'utilisateur
     * @param array $sites Tableau des sites à assigner [['code' => 'S01', 'nom' => 'Site 1'], ...]
     * @return bool True si succès
     */
    public static function assignSitesToUser($userId, $sites) {
        try {
            $pdo = DatabaseConfig::getConnection();

            // Valider le format des sites
            foreach ($sites as $site) {
                if (!isset($site['code']) || !isset($site['nom'])) {
                    throw new Exception("Format de site invalide. Chaque site doit avoir 'code' et 'nom'");
                }
            }

            // Encoder en JSON
            $sitesJson = json_encode($sites, JSON_UNESCAPED_UNICODE);

            // Mettre à jour l'utilisateur
            $stmt = $pdo->prepare("UPDATE user SET sites_acces = ? WHERE NUt = ?");
            $result = $stmt->execute([$sitesJson, $userId]);

            if ($result) {
                Utils::logError("Sites assignés avec succès à l'utilisateur $userId", [
                    'sites_count' => count($sites)
                ]);
            }

            return $result;

        } catch (Exception $e) {
            Utils::logError("Erreur lors de l'assignation des sites", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Récupérer tous les sites disponibles dans la base
     *
     * @return array Liste de tous les sites distincts
     */
    public static function getAllAvailableSites() {
        try {
            $pdo = DatabaseConfig::getConnection();

            $stmt = $pdo->query("
                SELECT DISTINCT CodeSite as code, NomSite as nom
                FROM pesee
                WHERE CodeSite IS NOT NULL
                  AND CodeSite != ''
                  AND NomSite IS NOT NULL
                  AND NomSite != ''
                ORDER BY NomSite
            ");

            return $stmt->fetchAll();

        } catch (PDOException $e) {
            Utils::logError("Erreur lors de la récupération des sites disponibles", [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Supprimer tous les sites d'un utilisateur (accès illimité)
     *
     * @param int $userId ID de l'utilisateur
     * @return bool True si succès
     */
    public static function removeAllSites($userId) {
        try {
            $pdo = DatabaseConfig::getConnection();

            $stmt = $pdo->prepare("UPDATE user SET sites_acces = NULL WHERE NUt = ?");
            $result = $stmt->execute([$userId]);

            if ($result) {
                Utils::logError("Tous les sites ont été retirés pour l'utilisateur $userId (accès illimité)");
            }

            return $result;

        } catch (PDOException $e) {
            Utils::logError("Erreur lors de la suppression des sites", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Vérifier si un utilisateur peut accéder à des données
     * Utilise le JWT pour récupérer l'utilisateur et vérifier les permissions
     *
     * @param array $requiredSiteCodes Codes des sites requis (optionnel)
     * @return array Payload JWT avec les sites de l'utilisateur
     */
    public static function requireSiteAccess($requiredSiteCodes = []) {
        try {
            // Vérifier le token JWT
            $payload = JwtManager::requireAuth();

            // Si aucun site requis, juste retourner le payload
            if (empty($requiredSiteCodes)) {
                $payload['user_sites'] = self::getUserSites($payload['user_id']);
                return $payload;
            }

            // Vérifier l'accès aux sites requis
            $userSites = self::getUserSites($payload['user_id']);
            $userSiteCodes = array_column($userSites, 'code');

            // Si l'utilisateur n'a pas de restriction, autoriser
            if (empty($userSiteCodes)) {
                $payload['user_sites'] = [];
                return $payload;
            }

            // Vérifier que l'utilisateur a accès à au moins un des sites requis
            $hasAccess = false;
            foreach ($requiredSiteCodes as $siteCode) {
                if (in_array($siteCode, $userSiteCodes)) {
                    $hasAccess = true;
                    break;
                }
            }

            if (!$hasAccess) {
                Utils::errorResponse('Accès refusé à ce site', 403);
            }

            $payload['user_sites'] = $userSites;
            return $payload;

        } catch (Exception $e) {
            Utils::errorResponse('Erreur de vérification d\'accès: ' . $e->getMessage(), 403);
        }
    }

    /**
     * Obtenir des statistiques sur l'utilisation des sites par utilisateur
     *
     * @return array Statistiques
     */
    public static function getSiteStatistics() {
        try {
            $pdo = DatabaseConfig::getConnection();

            // Nombre d'utilisateurs par restriction de sites
            $stmt = $pdo->query("
                SELECT
                    COUNT(*) as total_users,
                    SUM(CASE WHEN sites_acces IS NULL OR sites_acces = '[]' THEN 1 ELSE 0 END) as unrestricted_users,
                    SUM(CASE WHEN sites_acces IS NOT NULL AND sites_acces != '[]' THEN 1 ELSE 0 END) as restricted_users
                FROM user
                WHERE actif = 1
            ");

            $stats = $stmt->fetch();

            // Liste des sites les plus utilisés
            $stmt = $pdo->query("
                SELECT CodeSite, NomSite, COUNT(*) as pesee_count
                FROM pesee
                WHERE CodeSite IS NOT NULL AND CodeSite != ''
                GROUP BY CodeSite, NomSite
                ORDER BY pesee_count DESC
                LIMIT 10
            ");

            $stats['top_sites'] = $stmt->fetchAll();

            return $stats;

        } catch (PDOException $e) {
            Utils::logError("Erreur lors de la récupération des statistiques", [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
}
?>
