// pdf-export-separated.js - Export PDF en deux fichiers séparés

/**
 * Export PDF séparé - Graphiques en portrait
 */
async function exportGraphiquesToPDF() {
    try {
        showTemporaryMessage('📊 Génération PDF Graphiques...', 'warning');
        
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('portrait', 'mm', 'a4');
        
        // Configuration UTF-8
        pdf.setFont("helvetica");
        pdf.setCharSpace(0);
        
        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();
        const margin = 15;
        let currentY = margin;
        
        // === EN-TÊTE ===
        pdf.setFontSize(24);
        pdf.setTextColor(102, 126, 234);
        pdf.text('KOMBAR FERD', margin, currentY);
        currentY += 10;
        
        pdf.setFontSize(18);
        pdf.text('Rapport Graphiques - Analyse des Pesees', margin, currentY);
        currentY += 15;
        
        // Date et heure
        pdf.setFontSize(12);
        pdf.setTextColor(80, 80, 80);
        const now = new Date();
        pdf.text(`Genere le ${now.toLocaleDateString('fr-FR')} a ${now.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}`, margin, currentY);
        currentY += 15;
        
        // === FILTRES APPLIQUÉS ===
        pdf.setDrawColor(200, 200, 200);
        pdf.setFillColor(248, 249, 250);
        pdf.rect(margin, currentY, pageWidth - 2*margin, 30, 'FD');
        currentY += 8;
        
        pdf.setFontSize(14);
        pdf.setTextColor(0, 0, 0);
        pdf.text('Criteres de Filtrage Appliques', margin + 5, currentY);
        currentY += 8;
        
        pdf.setFontSize(10);
        const filters = reportsData.currentFilters;
        
        const periodText = filters.period === 'custom' 
            ? `Periode personnalisee: ${filters.startDate} vers ${filters.endDate}`
            : `Periode: ${filters.period}`;
        pdf.text(`- ${periodText}`, margin + 5, currentY);
        currentY += 5;
        
        if (filters.site !== 'all') {
            pdf.text(`- Site: ${filters.site}`, margin + 5, currentY);
            currentY += 5;
        }
        if (filters.client !== 'all') {
            pdf.text(`- Client: ${filters.client}`, margin + 5, currentY);
            currentY += 5;
        }
        if (filters.fournisseur !== 'all') {
            pdf.text(`- Fournisseur: ${filters.fournisseur}`, margin + 5, currentY);
            currentY += 5;
        }
        
        currentY += 15;
        
        // === MÉTRIQUES CLÉS ===
        pdf.setFontSize(16);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Indicateurs Cles de Performance', margin, currentY);
        currentY += 12;
        
        const metrics = {
            totalWeighings: document.getElementById('totalWeighingsMetric')?.textContent || '0',
            totalWeight: document.getElementById('totalWeightMetric')?.textContent || '0 T',
            avgWeight: document.getElementById('avgWeightMetric')?.textContent || '0 kg',
            activeClients: document.getElementById('activeClientsMetric')?.textContent || '0',
            activeFournisseurs: document.getElementById('activeFournisseursMetric')?.textContent || '0'
        };
        
        // Métriques en tableau
        pdf.setFillColor(102, 126, 234);
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(10);
        
        const metricsData = [
            ['Indicateur', 'Valeur'],
            ['Total des pesees', metrics.totalWeighings],
            ['Poids total', metrics.totalWeight],
            ['Poids moyen', metrics.avgWeight],
            ['Clients actifs', metrics.activeClients],
            ['Fournisseurs actifs', metrics.activeFournisseurs]
        ];
        
        const colWidthsMetrics = [80, 60];
        let startX = margin;
        
        metricsData.forEach((row, rowIndex) => {
            if (rowIndex === 0) {
                // En-tête
                pdf.setFillColor(102, 126, 234);
                pdf.setTextColor(255, 255, 255);
                pdf.rect(startX, currentY, colWidthsMetrics[0] + colWidthsMetrics[1], 8, 'F');
            } else {
                // Lignes alternées
                if (rowIndex % 2 === 0) {
                    pdf.setFillColor(248, 249, 250);
                    pdf.rect(startX, currentY, colWidthsMetrics[0] + colWidthsMetrics[1], 6, 'F');
                }
                pdf.setTextColor(0, 0, 0);
            }
            
            let cellX = startX + 3;
            row.forEach((cell, cellIndex) => {
                pdf.text(cell.toString(), cellX, currentY + (rowIndex === 0 ? 6 : 4));
                cellX += colWidthsMetrics[cellIndex];
            });
            
            currentY += rowIndex === 0 ? 10 : 6;
        });
        
        currentY += 15;
        
        // === GRAPHIQUES HAUTE QUALITÉ ===
        const chartsToExport = [
            { id: 'evolutionChart', title: 'Evolution des Pesees dans le Temps', fullPage: true },
            { id: 'productsChart', title: 'Repartition par Type de Produit', fullPage: false },
            { id: 'topClientsChart', title: 'Top 10 des Clients Principaux', fullPage: false },
            { id: 'topFournisseursChart', title: 'Top 10 des Fournisseurs Principaux', fullPage: false },
            { id: 'sitesChart', title: 'Performance par Site de Pesage', fullPage: false },
            { id: 'clientFournisseurChart', title: 'Relations Client-Fournisseur', fullPage: false }
        ];
        
        for (const chartInfo of chartsToExport) {
            const chartElement = document.getElementById(chartInfo.id);
            if (!chartElement) {
                console.warn(`Graphique ${chartInfo.id} non trouvé`);
                continue;
            }
            
            try {
                // Nouvelle page si nécessaire (sauf premier graphique)
                if (chartInfo.fullPage || currentY > pageHeight - 120) {
                    pdf.addPage('portrait');
                    currentY = margin;
                }
                
                // Titre du graphique
                pdf.setFontSize(14);
                pdf.setTextColor(102, 126, 234);
                pdf.text(chartInfo.title, margin, currentY);
                currentY += 12;
                
                // Capturer le conteneur complet avec légendes
                const chartContainer = chartElement.closest('.chart-card');
                if (!chartContainer) {
                    console.warn(`Container non trouvé pour ${chartInfo.id}`);
                    continue;
                }
                
                // Configuration haute résolution pour PDF
                const canvas = await html2canvas(chartContainer, {
                    backgroundColor: '#ffffff',
                    scale: 3, // Haute résolution
                    useCORS: true,
                    allowTaint: false,
                    logging: false,
                    onclone: (clonedDoc) => {
                        // Améliorer lisibilité des légendes
                        const legends = clonedDoc.querySelectorAll('.chartjs-legend, canvas + div, .chart-legend, .legend');
                        legends.forEach(legend => {
                            legend.style.fontSize = '16px';
                            legend.style.fontWeight = 'bold';
                            legend.style.color = '#000000';
                            legend.style.background = '#ffffff';
                        });
                    }
                });
                
                const imgData = canvas.toDataURL('image/png', 0.95);
                
                // Dimensions optimales selon le type
                const maxWidth = pageWidth - 2 * margin;
                const maxHeight = chartInfo.fullPage ? 180 : 120;
                
                let imgWidth = maxWidth;
                let imgHeight = (canvas.height * imgWidth) / canvas.width;
                
                if (imgHeight > maxHeight) {
                    imgHeight = maxHeight;
                    imgWidth = (canvas.width * imgHeight) / canvas.height;
                }
                
                // Centrer le graphique
                const imgX = (pageWidth - imgWidth) / 2;
                
                // Vérifier l'espace restant
                if (currentY + imgHeight > pageHeight - margin) {
                    pdf.addPage('portrait');
                    currentY = margin;
                    
                    // Répéter le titre
                    pdf.setFontSize(14);
                    pdf.setTextColor(102, 126, 234);
                    pdf.text(chartInfo.title, margin, currentY);
                    currentY += 12;
                }
                
                pdf.addImage(imgData, 'PNG', imgX, currentY, imgWidth, imgHeight);
                currentY += imgHeight + 20;
                
                console.log(`✅ Graphique ${chartInfo.title} ajouté au PDF`);
                
            } catch (error) {
                console.warn(`Erreur capture ${chartInfo.id}:`, error);
                
                // Message d'erreur dans le PDF
                pdf.setFontSize(12);
                pdf.setTextColor(200, 0, 0);
                pdf.text(`Attention: Impossible de capturer le graphique "${chartInfo.title}"`, margin, currentY);
                currentY += 15;
            }
        }
        
        // === PIED DE PAGE ===
        addPageNumbersToDocument(pdf, 'Rapport Graphiques');
        
        // Télécharger PDF Graphiques
        const fileName = `KOMBAR_FERD_Graphiques_${new Date().toISOString().split('T')[0]}_${Date.now()}.pdf`;
        pdf.save(fileName);
        
        showTemporaryMessage('✅ PDF Graphiques généré', 'success');
        console.log('📊 Export PDF Graphiques réussi:', fileName);
        
        return fileName;
        
    } catch (error) {
        console.error('❌ Erreur export PDF Graphiques:', error);
        showTemporaryMessage('❌ Erreur génération PDF Graphiques', 'error');
        throw error;
    }
}

/**
 * Export PDF séparé - Tableau en paysage
 */
async function exportTableauToPDF() {
    try {
        showTemporaryMessage('📋 Génération PDF Tableau...', 'warning');
        
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('landscape', 'mm', 'a4'); // Format paysage forcé
        
        // Configuration UTF-8
        pdf.setFont("helvetica");
        pdf.setCharSpace(0);
        
        const pageWidth = pdf.internal.pageSize.getWidth(); // 297mm
        const pageHeight = pdf.internal.pageSize.getHeight(); // 210mm
        const margin = 15;
        let currentY = margin;
        
        // === EN-TÊTE PAYSAGE ===
        pdf.setFontSize(20);
        pdf.setTextColor(102, 126, 234);
        pdf.text('KOMBAR FERD - Donnees Detaillees des Pesees', margin, currentY);
        currentY += 12;
        
        pdf.setFontSize(12);
        pdf.setTextColor(80, 80, 80);
        const now = new Date();
        pdf.text(`Genere le ${now.toLocaleDateString('fr-FR')} a ${now.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}`, margin, currentY);
        currentY += 8;
        
        // Filtres en ligne
        pdf.setFontSize(10);
        const filters = reportsData.currentFilters;
        
        let filtersText = `Filtres appliques: `;
        filtersText += filters.period === 'custom' 
            ? `Periode: ${filters.startDate} vers ${filters.endDate}` 
            : `Periode: ${filters.period}`;
        
        if (filters.site !== 'all') filtersText += ` | Site: ${filters.site}`;
        if (filters.client !== 'all') filtersText += ` | Client: ${filters.client}`;
        if (filters.fournisseur !== 'all') filtersText += ` | Fournisseur: ${filters.fournisseur}`;
        
        pdf.text(filtersText, margin, currentY);
        currentY += 15;
        
        // === RÉCUPÉRER DONNÉES ===
        const data = await loadReportsData();
        if (!data || data.length === 0) {
            pdf.setFontSize(14);
            pdf.setTextColor(150, 150, 150);
            pdf.text('Aucune donnee disponible pour la periode et les criteres selectionnes.', margin, currentY);
            
            const fileName = `KOMBAR_FERD_Tableau_Vide_${new Date().toISOString().split('T')[0]}.pdf`;
            pdf.save(fileName);
            
            showTemporaryMessage('⚠️ PDF Tableau vide généré', 'warning');
            return fileName;
        }
        
        // === RÉSUMÉ DES DONNÉES ===
        pdf.setFontSize(12);
        pdf.setTextColor(0, 0, 0);
        const totalRecords = data.length;
        const displayedRecords = Math.min(1000, totalRecords); // Limite à 1000 lignes
        
        pdf.text(`Total des enregistrements: ${totalRecords} | Affiches: ${displayedRecords}`, margin, currentY);
        currentY += 12;
        
        // === TABLEAU OPTIMISÉ PAYSAGE ===
        // Colonnes avec largeurs optimisées pour A4 paysage
        const colWidths = [32, 40, 55, 55, 50, 40, 35]; // Total: 307mm
        const headers = ['Date', 'Code Pesee', 'Client', 'Fournisseur', 'Produit', 'Site', 'Poids (kg)'];
        
        // En-têtes avec style professionnel
        pdf.setFillColor(102, 126, 234);
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(11);
        
        const totalWidth = colWidths.reduce((sum, width) => sum + width, 0);
        const tableStartX = (pageWidth - totalWidth) / 2; // Centrer le tableau
        
        pdf.rect(tableStartX, currentY, totalWidth, 12, 'F');
        
        let startX = tableStartX + 3;
        headers.forEach((header, index) => {
            pdf.text(header, startX, currentY + 8);
            startX += colWidths[index];
        });
        currentY += 16;
        
        // === DONNÉES DU TABLEAU ===
        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(9);
        
        const maxRowsPerPage = Math.floor((pageHeight - currentY - 20) / 7);
        let rowCount = 0;
        let pageNum = 1;
        
        for (let i = 0; i < Math.min(displayedRecords, data.length); i++) {
            const item = data[i];
            
            // Nouvelle page si nécessaire
            if (rowCount >= maxRowsPerPage) {
                // Numéro de page
                pdf.setFontSize(8);
                pdf.setTextColor(120, 120, 120);
                pdf.text(`Page ${pageNum}`, pageWidth - 30, pageHeight - 10);
                
                pdf.addPage('landscape');
                pageNum++;
                currentY = margin;
                
                // Répéter en-têtes
                pdf.setFillColor(102, 126, 234);
                pdf.setTextColor(255, 255, 255);
                pdf.setFontSize(11);
                pdf.rect(tableStartX, currentY, totalWidth, 12, 'F');
                
                startX = tableStartX + 3;
                headers.forEach((header, index) => {
                    pdf.text(header, startX, currentY + 8);
                    startX += colWidths[index];
                });
                currentY += 16;
                
                pdf.setTextColor(0, 0, 0);
                pdf.setFontSize(9);
                rowCount = 0;
            }
            
            // Alternance de couleurs
            if (rowCount % 2 === 0) {
                pdf.setFillColor(248, 249, 250);
                pdf.rect(tableStartX, currentY - 1, totalWidth, 7, 'F');
            }
            
            // Données de la ligne
            const rowData = [
                new Date(item.dmv).toLocaleDateString('fr-FR'),
                item.CodePesee || '-',
                item.NomClient || '-',
                item.NomFournisseur || '-',
                item.NomProduit || '-',
                item.NomSite || item.CodeSite || '-',
                `${parseFloat(item.PoidsNet || 0).toLocaleString()}`
            ];
            
            startX = tableStartX + 3;
            rowData.forEach((cellData, index) => {
                let text = cellData.toString();
                
                // Troncature intelligente selon la largeur
                const maxLengths = [12, 18, 25, 25, 22, 18, 12];
                const maxLen = maxLengths[index];
                
                if (text.length > maxLen) {
                    text = text.substring(0, maxLen - 2) + '..';
                }
                
                pdf.text(text, startX, currentY + 5);
                startX += colWidths[index];
            });
            
            currentY += 7;
            rowCount++;
        }
        
        // === PIED DE DERNIÈRE PAGE ===
        if (totalRecords > displayedRecords) {
            currentY += 10;
            pdf.setFontSize(10);
            pdf.setTextColor(150, 150, 150);
            pdf.text(`Note: Seules les ${displayedRecords} premieres lignes sont affichees sur ${totalRecords} au total.`, margin, currentY);
        }
        
        // Numéro de dernière page
        pdf.setFontSize(8);
        pdf.setTextColor(120, 120, 120);
        pdf.text(`Page ${pageNum}`, pageWidth - 30, pageHeight - 10);
        
        // Télécharger PDF Tableau
        const fileName = `KOMBAR_FERD_Tableau_${new Date().toISOString().split('T')[0]}_${Date.now()}.pdf`;
        pdf.save(fileName);
        
        showTemporaryMessage('✅ PDF Tableau généré', 'success');
        console.log('📋 Export PDF Tableau réussi:', fileName);
        
        return fileName;
        
    } catch (error) {
        console.error('❌ Erreur export PDF Tableau:', error);
        showTemporaryMessage('❌ Erreur génération PDF Tableau', 'error');
        throw error;
    }
}

/**
 * Export PDF complet - Les deux fichiers
 */
async function exportBothPDFs() {
    try {
        showTemporaryMessage('📁 Génération des deux PDF...', 'info');
        
        // Générer les deux PDF en parallèle
        const [graphiquesFile, tableauFile] = await Promise.all([
            exportGraphiquesToPDF(),
            exportTableauToPDF()
        ]);
        
        showTemporaryMessage('✅ Deux PDF générés avec succès', 'success');
        console.log('📁 Export complet réussi:', { graphiquesFile, tableauFile });
        
        return { graphiquesFile, tableauFile };
        
    } catch (error) {
        console.error('❌ Erreur export PDF complet:', error);
        showTemporaryMessage('❌ Erreur génération PDF complet', 'error');
    }
}

/**
 * Ajouter numéros de page à un document
 */
function addPageNumbersToDocument(pdf, documentTitle) {
    const totalPages = pdf.internal.getNumberOfPages();
    
    for (let i = 1; i <= totalPages; i++) {
        pdf.setPage(i);
        pdf.setFontSize(8);
        pdf.setTextColor(120, 120, 120);
        
        // Ligne de séparation
        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();
        
        pdf.setDrawColor(220, 220, 220);
        pdf.line(15, pageHeight - 15, pageWidth - 15, pageHeight - 15);
        
        // Titre et numéro de page
        pdf.text(documentTitle, 15, pageHeight - 8);
        
        const pageText = `Page ${i} / ${totalPages}`;
        const textWidth = pdf.getTextWidth(pageText);
        pdf.text(pageText, pageWidth - 15 - textWidth, pageHeight - 8);
    }
}

// Export pour utilisation globale
if (typeof window !== 'undefined') {
    window.exportGraphiquesToPDF = exportGraphiquesToPDF;
    window.exportTableauToPDF = exportTableauToPDF;
    window.exportBothPDFs = exportBothPDFs;
}