<?php
// test_jwt_system.php - Test complet du système JWT
echo "=== TEST DU SYSTÈME JWT ===\n";

try {
    require_once 'config.php';
    require_once 'JwtManager.php';
    
    echo "✅ Classes JWT chargées\n";
    
    // Test 1: Création d'un token
    echo "\n=== TEST 1: CRÉATION DE TOKEN ===\n";
    
    $payload = [
        'user_id' => 1,
        'username' => 'ADMIN',
        'nom_prenoms' => 'Administrateur',
        'groupe' => 'admin'
    ];
    
    $token = JwtManager::createToken($payload);
    echo "✅ Token créé: " . substr($token, 0, 50) . "...\n";
    echo "✅ Longueur du token: " . strlen($token) . " caractères\n";
    
    // Test 2: Vérification du token
    echo "\n=== TEST 2: VÉRIFICATION DE TOKEN ===\n";
    
    $decodedPayload = JwtManager::verifyToken($token);
    echo "✅ Token vérifié avec succès\n";
    echo "✅ Données décodées:\n";
    foreach ($decodedPayload as $key => $value) {
        echo "  - $key: $value\n";
    }
    
    // Test 3: Token expiré
    echo "\n=== TEST 3: TOKEN EXPIRÉ ===\n";
    
    $expiredToken = JwtManager::createToken($payload, time() - 1); // Expiré il y a 1 seconde
    try {
        JwtManager::verifyToken($expiredToken);
        echo "❌ Le token expiré n'a pas été détecté!\n";
    } catch (Exception $e) {
        echo "✅ Token expiré correctement détecté: " . $e->getMessage() . "\n";
    }
    
    // Test 4: Token invalide
    echo "\n=== TEST 4: TOKEN INVALIDE ===\n";
    
    $invalidToken = "invalid.token.here";
    try {
        JwtManager::verifyToken($invalidToken);
        echo "❌ Le token invalide n'a pas été détecté!\n";
    } catch (Exception $e) {
        echo "✅ Token invalide correctement détecté: " . $e->getMessage() . "\n";
    }
    
    // Test 5: Refresh token
    echo "\n=== TEST 5: REFRESH TOKEN ===\n";
    
    $refreshToken = JwtManager::createRefreshToken($payload);
    echo "✅ Refresh token créé: " . substr($refreshToken, 0, 50) . "...\n";
    
    $refreshPayload = JwtManager::verifyToken($refreshToken);
    echo "✅ Type du refresh token: " . ($refreshPayload['type'] ?? 'non défini') . "\n";
    
    // Test 6: Extraction depuis header
    echo "\n=== TEST 6: EXTRACTION DEPUIS HEADER ===\n";
    
    // Simuler un header Authorization
    $_SERVER['HTTP_AUTHORIZATION'] = "Bearer $token";
    $extractedToken = JwtManager::extractTokenFromHeader();
    
    if ($extractedToken === $token) {
        echo "✅ Token extrait correctement du header\n";
    } else {
        echo "❌ Échec d'extraction du token du header\n";
    }
    
    // Test 7: Vérification de proximité d'expiration
    echo "\n=== TEST 7: PROXIMITÉ D'EXPIRATION ===\n";
    
    $shortToken = JwtManager::createToken($payload, time() + 120); // Expire dans 2 minutes
    $isNearExpiration = JwtManager::isTokenNearExpiration($shortToken, 300); // Seuil 5 minutes
    
    echo "✅ Token proche de l'expiration: " . ($isNearExpiration ? 'OUI' : 'NON') . "\n";
    
    // Test 8: Blacklist (simulation sans base de données)
    echo "\n=== TEST 8: BLACKLIST (SIMULATION) ===\n";
    
    try {
        // Ce test échouera car pas de DB, mais on peut voir la logique
        JwtManager::blacklistToken($token);
        echo "ℹ️  Blacklist testé (nécessite une base de données)\n";
    } catch (Exception $e) {
        echo "ℹ️  Blacklist non testable sans DB: " . $e->getMessage() . "\n";
    }
    
    echo "\n=== RÉSUMÉ DES TESTS JWT ===\n";
    echo "✅ Création de tokens: OK\n";
    echo "✅ Vérification de tokens: OK\n";
    echo "✅ Détection d'expiration: OK\n";
    echo "✅ Détection de tokens invalides: OK\n";
    echo "✅ Refresh tokens: OK\n";
    echo "✅ Extraction des headers: OK\n";
    echo "✅ Proximité d'expiration: OK\n";
    echo "ℹ️  Blacklist: Nécessite une base de données\n";
    
    // Test 9: Test d'authentification complète (simulation)
    echo "\n=== TEST 9: SIMULATION AUTH COMPLÈTE ===\n";
    
    require_once 'auth.php';
    $authService = new AuthService();
    
    // Simuler une vérification de token
    try {
        $userInfo = $authService->verifyToken($token);
        if ($userInfo) {
            echo "✅ Vérification d'authentification: SUCCÈS\n";
            echo "✅ Token JWT valide: " . ($userInfo['token_valid'] ? 'OUI' : 'NON') . "\n";
        } else {
            echo "ℹ️  Vérification d'authentification nécessite une base de données\n";
        }
    } catch (Exception $e) {
        echo "ℹ️  Test d'auth complet nécessite une DB: " . $e->getMessage() . "\n";
    }
    
    echo "\n🎉 SYSTÈME JWT IMPLÉMENTÉ AVEC SUCCÈS!\n";
    
} catch (Exception $e) {
    echo "❌ Erreur durante les tests: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>