// pdf-export-enhanced.js - Export PDF amélioré avec légendes et tableau complet

/**
 * Fonction améliorée d'export PDF avec légendes et tableau complet
 */
async function exportToPDFEnhanced() {
    try {
        showTemporaryMessage('📄 Génération du PDF en cours...', 'warning');
        
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('p', 'mm', 'a4');
        
        // Configuration UTF-8
        pdf.setFont("helvetica");
        pdf.setCharSpace(0);
        
        // Configuration
        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();
        const margin = 15;
        let currentY = margin;
        
        // === EN-TÊTE AMÉLIORÉ ===
        // Logo et titre principal (sans emojis pour éviter problèmes encodage)
        pdf.setFontSize(22);
        pdf.setTextColor(102, 126, 234);
        pdf.text('OZIXSOFT', margin, currentY);
        currentY += 8;
        
        pdf.setFontSize(18);
        pdf.text('Rapport de Pesees Detaille', margin, currentY);
        currentY += 12;
        
        // Informations du rapport
        pdf.setFontSize(11);
        pdf.setTextColor(80, 80, 80);
        const now = new Date();
        pdf.text(`Genere le ${now.toLocaleDateString('fr-FR')} a ${now.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}`, margin, currentY);
        currentY += 6;
        
        // Filtres appliqués - Section encadrée
        pdf.setDrawColor(200, 200, 200);
        pdf.rect(margin, currentY, pageWidth - 2*margin, 25);
        currentY += 6;
        
        pdf.setFontSize(12);
        pdf.setTextColor(0, 0, 0);
        pdf.text('Criteres de Filtrage:', margin + 3, currentY);
        currentY += 6;
        
        pdf.setFontSize(10);
        const filters = reportsData.currentFilters;
        
        // Période
        const periodText = filters.period === 'custom' 
            ? `Periode personnalisee: ${filters.startDate} vers ${filters.endDate}`
            : `Periode: ${filters.period}`;
        pdf.text(`- ${periodText}`, margin + 3, currentY);
        currentY += 4;
        
        // Autres filtres
        if (filters.site !== 'all') {
            pdf.text(`- Site: ${filters.site}`, margin + 3, currentY);
            currentY += 4;
        }
        if (filters.client !== 'all') {
            pdf.text(`- Client: ${filters.client}`, margin + 3, currentY);
            currentY += 4;
        }
        if (filters.fournisseur !== 'all') {
            pdf.text(`- Fournisseur: ${filters.fournisseur}`, margin + 3, currentY);
            currentY += 4;
        }
        
        currentY += 8;
        
        // === MÉTRIQUES CLÉS EN COLONNES ===
        pdf.setFontSize(16);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Metriques Cles', margin, currentY);
        currentY += 10;
        
        // Récupérer les métriques
        const metrics = {
            totalWeighings: document.getElementById('totalWeighingsMetric')?.textContent || '0',
            totalWeight: document.getElementById('totalWeightMetric')?.textContent || '0 T',
            avgWeight: document.getElementById('avgWeightMetric')?.textContent || '0 kg',
            activeClients: document.getElementById('activeClientsMetric')?.textContent || '0',
            activeFournisseurs: document.getElementById('activeFournisseursMetric')?.textContent || '0'
        };
        
        // Affichage en 2 colonnes
        pdf.setFontSize(11);
        pdf.setTextColor(0, 0, 0);
        const col1X = margin;
        const col2X = pageWidth / 2 + 10;
        
        pdf.text(`Total des pesees: ${metrics.totalWeighings}`, col1X, currentY);
        pdf.text(`Clients actifs: ${metrics.activeClients}`, col2X, currentY);
        currentY += 6;
        
        pdf.text(`Poids total: ${metrics.totalWeight}`, col1X, currentY);
        pdf.text(`Fournisseurs actifs: ${metrics.activeFournisseurs}`, col2X, currentY);
        currentY += 6;
        
        pdf.text(`Poids moyen: ${metrics.avgWeight}`, col1X, currentY);
        currentY += 15;
        
        // === GRAPHIQUES AVEC LÉGENDES ===
        const chartsToExport = [
            { id: 'evolutionChart', title: 'Evolution des Pesees', full: true },
            { id: 'productsChart', title: 'Repartition par Produit', full: false },
            { id: 'topClientsChart', title: 'Top 10 Clients', full: false },
            { id: 'topFournisseursChart', title: 'Top 10 Fournisseurs', full: false },
            { id: 'sitesChart', title: 'Performance par Site', full: false },
            { id: 'clientFournisseurChart', title: 'Relations Client-Fournisseur', full: false }
        ];
        
        for (const chartInfo of chartsToExport) {
            const chartElement = document.getElementById(chartInfo.id);
            if (!chartElement) continue;
            
            try {
                // Nouvelle page si nécessaire
                if (currentY > pageHeight - 120) {
                    pdf.addPage();
                    currentY = margin;
                }
                
                // Titre du graphique
                pdf.setFontSize(14);
                pdf.setTextColor(102, 126, 234);
                pdf.text(chartInfo.title, margin, currentY);
                currentY += 10;
                
                // Capturer le conteneur complet avec légendes
                const chartContainer = chartElement.closest('.chart-card');
                if (!chartContainer) {
                    console.warn(`Container non trouvé pour ${chartInfo.id}`);
                    continue;
                }
                
                // Configuration avancée de html2canvas pour légendes lisibles
                const canvas = await html2canvas(chartContainer, {
                    backgroundColor: '#ffffff',
                    scale: 3, // Augmenter la résolution pour légendes plus nettes
                    useCORS: true,
                    allowTaint: false,
                    logging: false,
                    width: chartContainer.offsetWidth,
                    height: chartContainer.offsetHeight,
                    scrollX: 0,
                    scrollY: 0,
                    onclone: (clonedDoc) => {
                        // Améliorer le contraste des légendes dans le clone
                        const legends = clonedDoc.querySelectorAll('.chartjs-legend, canvas + div, .chart-legend');
                        legends.forEach(legend => {
                            legend.style.fontSize = '14px';
                            legend.style.fontWeight = 'bold';
                            legend.style.color = '#000000';
                        });
                    }
                });
                
                const imgData = canvas.toDataURL('image/png', 0.95);
                
                // Calculer les dimensions optimales
                const maxWidth = pageWidth - 2 * margin;
                const maxHeight = chartInfo.full ? 100 : 80;
                
                let imgWidth = maxWidth;
                let imgHeight = (canvas.height * imgWidth) / canvas.width;
                
                // Ajuster si trop haut
                if (imgHeight > maxHeight) {
                    imgHeight = maxHeight;
                    imgWidth = (canvas.width * imgHeight) / canvas.height;
                }
                
                // Centrer horizontalement
                const imgX = (pageWidth - imgWidth) / 2;
                
                // Vérifier l'espace disponible
                if (currentY + imgHeight > pageHeight - margin) {
                    pdf.addPage();
                    currentY = margin;
                    // Répéter le titre sur la nouvelle page
                    pdf.setFontSize(14);
                    pdf.setTextColor(102, 126, 234);
                    pdf.text(chartInfo.title, margin, currentY);
                    currentY += 10;
                }
                
                pdf.addImage(imgData, 'PNG', imgX, currentY, imgWidth, imgHeight);
                currentY += imgHeight + 15;
                
            } catch (error) {
                console.warn(`Erreur capture ${chartInfo.id}:`, error);
                // Message d'erreur dans le PDF
                pdf.setFontSize(10);
                pdf.setTextColor(200, 0, 0);
                pdf.text(`Attention: Graphique ${chartInfo.title} non disponible`, margin, currentY);
                currentY += 10;
            }
        }
        
        // === TABLEAU DÉTAILLÉ COMPLET ===
        await addDetailedTableToPDF(pdf, margin, pageWidth, pageHeight);
        
        // === PIED DE PAGE ===
        addPageNumbers(pdf);
        
        // Télécharger
        const fileName = `KOMBAR_FERD_Rapport_${new Date().toISOString().split('T')[0]}_${Date.now()}.pdf`;
        pdf.save(fileName);
        
        showTemporaryMessage('✅ PDF généré avec succès', 'success');
        console.log('📄 Export PDF amélioré réussi:', fileName);
        
    } catch (error) {
        console.error('❌ Erreur export PDF:', error);
        showTemporaryMessage('❌ Erreur lors de l\'export PDF', 'error');
    }
}

/**
 * Ajouter le tableau détaillé complet au PDF en format paysage
 */
async function addDetailedTableToPDF(pdf, margin, pageWidth, pageHeight) {
    // Nouvelle page en format paysage avec la bonne syntaxe
    pdf.addPage([297, 210]); // [largeur, hauteur] en mm pour A4 paysage
    
    // Recalculer les dimensions en paysage
    const landscapeWidth = 297; // Largeur A4 paysage
    const landscapeHeight = 210; // Hauteur A4 paysage  
    let currentY = margin;
    
    // Titre de la section
    pdf.setFontSize(16);
    pdf.setTextColor(102, 126, 234);
    pdf.text('Donnees Detaillees', margin, currentY);
    currentY += 12;
    
    // Récupérer toutes les données (pas juste les 100 premières)
    const data = await loadReportsData();
    if (!data || data.length === 0) {
        pdf.setFontSize(12);
        pdf.setTextColor(150, 150, 150);
        pdf.text('Aucune donnee disponible pour la periode selectionnee.', margin, currentY);
        return;
    }
    
    // Configuration du tableau en paysage - colonnes plus larges
    const colWidths = [28, 32, 45, 45, 40, 35, 32];
    const headers = ['Date', 'Code', 'Client', 'Fournisseur', 'Produit', 'Site', 'Poids'];
    
    // En-têtes avec style
    pdf.setFillColor(102, 126, 234);
    pdf.setTextColor(255, 255, 255);
    pdf.setFontSize(10);
    pdf.rect(margin, currentY, landscapeWidth - 2*margin, 10, 'F');
    
    let startX = margin + 3;
    headers.forEach((header, index) => {
        pdf.text(header, startX, currentY + 7);
        startX += colWidths[index];
    });
    currentY += 15;
    
    // Données du tableau
    pdf.setTextColor(0, 0, 0);
    pdf.setFontSize(9);
    
    const maxRowsPerPage = Math.floor((landscapeHeight - currentY - 30) / 6);
    let rowCount = 0;
    
    for (const item of data.slice(0, 500)) { // Limiter à 500 lignes max
        if (rowCount >= maxRowsPerPage) {
            // Nouvelle page en paysage
            pdf.addPage([297, 210]);
            currentY = margin;
            
            // Répéter les en-têtes
            pdf.setFillColor(102, 126, 234);
            pdf.setTextColor(255, 255, 255);
            pdf.setFontSize(10);
            pdf.rect(margin, currentY, landscapeWidth - 2*margin, 10, 'F');
            
            startX = margin + 3;
            headers.forEach((header, index) => {
                pdf.text(header, startX, currentY + 7);
                startX += colWidths[index];
            });
            currentY += 15;
            
            pdf.setTextColor(0, 0, 0);
            pdf.setFontSize(9);
            rowCount = 0;
        }
        
        // Alternance de couleurs
        if (rowCount % 2 === 0) {
            pdf.setFillColor(248, 249, 250);
            pdf.rect(margin, currentY - 1, landscapeWidth - 2*margin, 6, 'F');
        }
        
        // Données de la ligne
        const rowData = [
            new Date(item.dmv).toLocaleDateString('fr-FR'),
            item.CodePesee || '-',
            item.NomClient || '-',
            item.NomFournisseur || '-',
            item.NomProduit || '-',
            item.NomSite || item.CodeSite || '-',
            `${parseFloat(item.PoidsNet || 0).toLocaleString()} kg`
        ];
        
        startX = margin + 3;
        rowData.forEach((cellData, index) => {
            let text = cellData.toString();
            // Largeurs plus généreuses en paysage
            const maxLen = index < 2 ? 12 : (index === 6 ? 15 : 20);
            if (text.length > maxLen) {
                text = text.substring(0, maxLen - 2) + '..';
            }
            pdf.text(text, startX, currentY + 4);
            startX += colWidths[index];
        });
        
        currentY += 6;
        rowCount++;
    }
    
    // Note si données tronquées
    if (data.length > 500) {
        currentY += 8;
        pdf.setFontSize(10);
        pdf.setTextColor(150, 150, 150);
        pdf.text(`Note: Affichage des 500 premieres lignes sur ${data.length} total.`, margin, currentY);
    }
}

/**
 * Ajouter les numéros de page
 */
function addPageNumbers(pdf) {
    const totalPages = pdf.internal.getNumberOfPages();
    
    for (let i = 1; i <= totalPages; i++) {
        pdf.setPage(i);
        pdf.setFontSize(8);
        pdf.setTextColor(120, 120, 120);
        
        const pageText = `Page ${i} / ${totalPages}`;
        const textWidth = pdf.getTextWidth(pageText);
        const x = pdf.internal.pageSize.getWidth() - 15 - textWidth;
        const y = pdf.internal.pageSize.getHeight() - 8;
        
        pdf.text(pageText, x, y);
        
        // Ligne de séparation
        pdf.setDrawColor(220, 220, 220);
        pdf.line(15, y - 5, pdf.internal.pageSize.getWidth() - 15, y - 5);
    }
}

// Export pour utilisation globale
if (typeof window !== 'undefined') {
    window.exportToPDFEnhanced = exportToPDFEnhanced;
}