// notification-watcher.js - Surveillance automatique des nouvelles pesées
// Système de notifications en temps réel pour KOMBAR FERD

class NotificationWatcher {
    
    constructor() {
        this.lastCheckTime = null;
        this.pollingInterval = null;
        this.pollingFrequency = 30000; // 30 secondes
        this.isActive = false;
        this.lastPeseeId = null;
        
        console.log('🔔 NotificationWatcher initialisé');
    }
    
    /**
     * Démarrer la surveillance des nouvelles pesées
     */
    start() {
        if (this.isActive) {
            console.log('⚠️ Surveillance déjà active');
            return;
        }
        
        console.log('🚀 Démarrage surveillance nouvelles pesées...');
        this.isActive = true;
        this.lastCheckTime = new Date();
        
        // Récupérer l'ID de la dernière pesée actuelle
        this.initializeLastPeseeId();
        
        // Démarrer le polling
        this.pollingInterval = setInterval(() => {
            this.checkForNewPesees();
        }, this.pollingFrequency);
        
        console.log(`✅ Surveillance active (vérification toutes les ${this.pollingFrequency/1000}s)`);
    }
    
    /**
     * Arrêter la surveillance
     */
    stop() {
        if (!this.isActive) {
            return;
        }
        
        console.log('🛑 Arrêt surveillance nouvelles pesées');
        this.isActive = false;
        
        if (this.pollingInterval) {
            clearInterval(this.pollingInterval);
            this.pollingInterval = null;
        }
    }
    
    /**
     * Initialiser l'ID de la dernière pesée
     */
    async initializeLastPeseeId() {
        try {
            const response = await apiRequest('pesees.php?action=list&limit=1');
            console.log('🔧 Initialisation - Réponse API:', response);
            
            if (response.ok && response.data && response.data.success) {
                // Tester différentes structures possibles
                let pesees = [];
                if (response.data.data && response.data.data.pesees) {
                    pesees = response.data.data.pesees;
                } else if (response.data.data && Array.isArray(response.data.data)) {
                    pesees = response.data.data;
                } else if (response.data.pesees) {
                    pesees = response.data.pesees;
                } else if (Array.isArray(response.data)) {
                    pesees = response.data;
                }
                
                if (pesees.length > 0) {
                    // Trouver le vrai ID maximum au lieu de prendre le premier
                    const maxId = Math.max(...pesees.map(p => parseInt(p.ID)));
                    this.lastPeseeId = maxId.toString();
                    console.log('📊 ID maximum trouvé et défini:', this.lastPeseeId);
                    console.log('📊 Détail pesées:', pesees.map(p => `ID:${p.ID}`).join(', '));
                } else {
                    console.warn('⚠️ Aucune pesée valide pour initialisation');
                    console.warn('⚠️ Pesées récupérées:', pesees);
                }
            } else {
                console.warn('⚠️ Réponse API invalide pour initialisation:', response);
            }
        } catch (error) {
            console.warn('⚠️ Erreur initialisation ID dernière pesée:', error);
        }
    }
    
    /**
     * Vérifier s'il y a de nouvelles pesées
     */
    async checkForNewPesees() {
        if (!this.isActive) return;
        
        try {
            const timestamp = new Date().toLocaleTimeString();
            console.log(`🔍 [${timestamp}] Vérification nouvelles pesées...`);
            
            // Test direct avec fetch pour bypasser apiRequest si nécessaire
            let response;
            try {
                response = await apiRequest('pesees.php?action=list&limit=5');
                console.log('📊 Réponse apiRequest:', response);
            } catch (apiError) {
                console.warn('⚠️ apiRequest échoué, test fetch direct:', apiError);
                const fetchResponse = await fetch('./pesees.php?action=list&limit=5');
                const fetchData = await fetchResponse.json();
                response = {
                    ok: fetchResponse.ok,
                    data: fetchData
                };
                console.log('📊 Réponse fetch direct:', response);
            }
            
            if (!response.ok || !response.data || !response.data.success) {
                console.warn('⚠️ Erreur récupération pesées:', response.error || 'Structure invalide');
                console.warn('⚠️ Détails debug:', {
                    ok: response.ok,
                    hasData: !!response.data,
                    success: response.data ? response.data.success : 'N/A'
                });
                return;
            }
            
            // Debug détaillé de la structure
            console.log('🔍 Analyse structure response.data:');
            console.log('   - Type:', typeof response.data);
            console.log('   - Keys:', Object.keys(response.data));
            console.log('   - data.data exists:', !!response.data.data);
            console.log('   - data.data type:', response.data.data ? typeof response.data.data : 'undefined');
            if (response.data.data) {
                console.log('   - data.data keys:', Object.keys(response.data.data));
                console.log('   - data.data.pesees exists:', !!response.data.data.pesees);
                if (response.data.data.pesees) {
                    console.log('   - data.data.pesees length:', response.data.data.pesees.length);
                }
            }
            
            // Tester différentes structures possibles
            let pesees = [];
            if (response.data.data && response.data.data.pesees && Array.isArray(response.data.data.pesees)) {
                pesees = response.data.data.pesees;
                console.log('✅ Structure trouvée: response.data.data.pesees');
            } else if (response.data.data && Array.isArray(response.data.data)) {
                pesees = response.data.data;
                console.log('✅ Structure trouvée: response.data.data (array)');
            } else if (response.data.pesees && Array.isArray(response.data.pesees)) {
                pesees = response.data.pesees;
                console.log('✅ Structure trouvée: response.data.pesees');
            } else if (Array.isArray(response.data)) {
                pesees = response.data;
                console.log('✅ Structure trouvée: response.data (array)');
            } else {
                console.error('❌ Aucune structure de pesées reconnue!');
                console.log('🔍 Structure complète:', JSON.stringify(response.data, null, 2));
            }
            
            console.log(`📋 Pesées finalement récupérées: ${pesees.length} éléments`);
            if (pesees.length > 0) {
                console.log('📊 Premier élément:', pesees[0]);
                console.log('📊 Type premier élément:', typeof pesees[0]);
                console.log('📊 Keys premier élément:', pesees[0] ? Object.keys(pesees[0]) : 'N/A');
            }
            
            if (pesees.length === 0) {
                console.log('⚠️ Aucune pesée trouvée');
                return;
            }
            
            // Trouver l'ID maximum au lieu du premier
            const maxId = Math.max(...pesees.map(p => parseInt(p.ID)));
            const latestPeseeId = maxId.toString();
            console.log(`🔄 ID dernière pesée: ${latestPeseeId}, ID précédent: ${this.lastPeseeId}`);
            
            // Si c'est la première vérification, pas de notification
            if (this.lastPeseeId === null) {
                this.lastPeseeId = latestPeseeId;
                console.log(`📌 Initialisation ID de référence: ${latestPeseeId}`);
                return;
            }
            
            // Vérifier s'il y a de nouvelles pesées - DEBUG détaillé
            const newPesees = [];
            console.log(`🔍 Comparaison IDs pour détection nouvelles pesées:`);
            console.log(`   - lastPeseeId actuel: ${this.lastPeseeId} (type: ${typeof this.lastPeseeId})`);
            console.log(`   - latestPeseeId reçu: ${latestPeseeId} (type: ${typeof latestPeseeId})`);
            
            for (let i = 0; i < pesees.length; i++) {
                const pesee = pesees[i];
                const peseeId = pesee.ID;
                console.log(`   - Pesée ${i+1}: ID=${peseeId} (${typeof peseeId}), parseInt=${parseInt(peseeId)}`);
                console.log(`   - Comparaison: ${parseInt(peseeId)} > ${parseInt(this.lastPeseeId)} = ${parseInt(peseeId) > parseInt(this.lastPeseeId)}`);
                
                if (parseInt(peseeId) > parseInt(this.lastPeseeId)) {
                    newPesees.push(pesee);
                    console.log(`🆕 ✅ Nouvelle pesée détectée: ID ${peseeId}, Code: ${pesee.CodePesee}`);
                } else {
                    console.log(`   - ❌ Pesée ${peseeId} pas nouvelle (≤ ${this.lastPeseeId})`);
                    break; // Les pesées sont triées par ID décroissant
                }
            }
            
            console.log(`🔍 Résultat détection: ${newPesees.length} nouvelles pesées`);
            
            if (newPesees.length > 0) {
                console.log('🎉 NOUVELLES PESÉES DÉTECTÉES:', newPesees.length);
                console.log('📋 Détail nouvelles pesées:', newPesees.map(p => `ID:${p.ID} Code:${p.CodePesee}`));
                
                // Mettre à jour le dernier ID
                const oldId = this.lastPeseeId;
                this.lastPeseeId = latestPeseeId;
                console.log(`📌 Mise à jour lastPeseeId: ${oldId} → ${this.lastPeseeId}`);
                
                // Envoyer les notifications
                console.log('🔔 Appel sendNewPeseeNotifications...');
                await this.sendNewPeseeNotifications(newPesees);
                
                // Mettre à jour l'interface si elle est visible
                console.log('🔄 Appel updateInterface...');
                this.updateInterface(newPesees);
            } else {
                console.log('😴 Aucune nouvelle pesée détectée');
            }
            
        } catch (error) {
            console.error('❌ Erreur vérification nouvelles pesées:', error);
        }
    }
    
    /**
     * Envoyer les notifications pour les nouvelles pesées
     */
    async sendNewPeseeNotifications(newPesees) {
        console.log('🔔 =================== DÉBUT ENVOI NOTIFICATIONS ===================');
        console.log(`🔔 Envoi notifications pour ${newPesees.length} nouvelles pesées...`);
        console.log('🔔 Détail pesées à notifier:', newPesees);
        
        // Si plus d'une nouvelle pesée, notification groupée
        if (newPesees.length > 1) {
            const title = `🆕 ${newPesees.length} Nouvelles Pesées`;
            const body = `Dernières pesées: ${newPesees.slice(0, 2).map(p => p.CodePesee).join(', ')}${newPesees.length > 2 ? '...' : ''}`;
            
            await this.sendNotification(title, body, {
                tag: 'nouvelles-pesees-groupe',
                data: {
                    type: 'nouvelles_pesees',
                    count: newPesees.length,
                    pesees: newPesees.map(p => p.ID)
                }
            });
        } else {
            // Notification pour une seule pesée
            const pesee = newPesees[0];
            const siteName = pesee.NomSite || pesee.Site || 'OZIXSOFT';
            const title = `🆕Nouvelle Pesée SITE:${siteName}`;
            const body = `${pesee.CodePesee} • ${pesee.NomClient || 'N/A'}\n${pesee.NomFournisseur || 'N/A'} • ${pesee.PoidsNet || 'N/A'} kg`;
            
            await this.sendNotification(title, body, {
                tag: `pesee-${pesee.ID}`,
                data: {
                    type: 'nouvelle_pesee',
                    peseeId: pesee.ID,
                    codePesee: pesee.CodePesee
                }
            });
        }
    }
    
    /**
     * Envoyer une notification
     */
    async sendNotification(title, body, options = {}) {
        console.log('📤 ================== DÉBUT SENDNOTIFICATION ==================');
        console.log('📤 Titre:', title);
        console.log('📤 Corps:', body);
        console.log('📤 Options:', options);
        
        try {
            // Vérifier si les notifications sont activées
            console.log('📤 Vérification notificationManager...');
            console.log('📤 window.notificationManager exists:', !!window.notificationManager);
            if (window.notificationManager) {
                console.log('📤 notificationManager.isEnabled:', window.notificationManager.isEnabled);
            }
            
            if (!window.notificationManager || !window.notificationManager.isEnabled) {
                console.log('📴 Notifications non activées, utilisation notification locale');
                this.showInAppNotification(title, body);
                return;
            }
            
            // Essayer d'envoyer via le système push
            const notificationData = {
                title: title,
                body: body,
                icon: './icons/icon-192x192.png',
                badge: './icons/icon-96x96.png',
                requireInteraction: false,
                actions: [{
                    action: 'view',
                    title: '👁️ Voir',
                    icon: './icons/icon-96x96.png'
                }],
                ...options
            };
            
            // Notification push si possible
            if ('serviceWorker' in navigator && 'PushManager' in window) {
                const registration = await navigator.serviceWorker.ready;
                await registration.showNotification(title, notificationData);
                console.log('✅ Notification push envoyée');
            } else {
                // Fallback notification navigateur
                if (Notification.permission === 'granted') {
                    new Notification(title, {
                        body: body,
                        icon: './icons/icon-192x192.png',
                        tag: options.tag || 'nouvelle-pesee'
                    });
                    console.log('✅ Notification navigateur envoyée');
                } else {
                    // Fallback notification in-app
                    this.showInAppNotification(title, body);
                }
            }
        } catch (error) {
            console.error('❌ Erreur envoi notification:', error);
            this.showInAppNotification(title, body);
        }
    }
    
    /**
     * Afficher une notification dans l'interface
     */
    showInAppNotification(title, body) {
        console.log('📱 ================= DÉBUT SHOWINAPPNOTIFICATION =================');
        console.log('📱 Titre:', title);
        console.log('📱 Corps:', body);
        console.log('📱 Document exists:', !!document);
        console.log('📱 Document.body exists:', !!document.body);
        
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 16px 20px;
            border-radius: 12px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            z-index: 10001;
            max-width: 350px;
            font-family: inherit;
            animation: slideInRight 0.4s ease-out;
            cursor: pointer;
        `;
        
        notification.innerHTML = `
            <div style="display: flex; align-items: flex-start; gap: 12px;">
                <div style="font-size: 1.2em;">🆕</div>
                <div style="flex: 1;">
                    <div style="font-weight: bold; margin-bottom: 4px; font-size: 15px;">${title}</div>
                    <div style="font-size: 13px; opacity: 0.9; line-height: 1.3;">${body}</div>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" style="
                    background: rgba(255,255,255,0.2);
                    border: none;
                    color: white;
                    border-radius: 50%;
                    width: 26px;
                    height: 26px;
                    cursor: pointer;
                    font-size: 14px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                ">×</button>
            </div>
        `;
        
        // Clic pour aller à l'onglet pesées
        notification.addEventListener('click', (e) => {
            if (e.target.tagName !== 'BUTTON') {
                if (typeof showSection === 'function') {
                    showSection('search');
                    if (typeof rechercher === 'function') {
                        rechercher();
                    }
                }
                notification.remove();
            }
        });
        
        console.log('📱 Ajout de la notification au DOM...');
        document.body.appendChild(notification);
        console.log('📱 Notification ajoutée au DOM');
        
        // Auto-supprimer après 8 secondes
        setTimeout(() => {
            console.log('📱 Auto-suppression notification après 8s...');
            if (notification.parentElement) {
                notification.remove();
                console.log('📱 Notification supprimée');
            }
        }, 8000);
        
        console.log('✅ Notification in-app affichée et programmée pour suppression');
    }
    
    /**
     * Mettre à jour l'interface avec les nouvelles pesées
     */
    updateInterface(newPesees) {
        console.log('🔄 Mise à jour interface avec nouvelles pesées');
        
        // Mettre à jour le badge notifications
        this.updateNotificationBadge(newPesees.length);
        
        // Si on est sur l'onglet recherche, actualiser automatiquement
        const currentSection = document.querySelector('.content-section:not([style*="display: none"])');
        if (currentSection && currentSection.id === 'search') {
            console.log('📊 Actualisation automatique onglet recherche');
            if (typeof rechercher === 'function') {
                setTimeout(rechercher, 1000); // Délai pour éviter les appels trop fréquents
            }
        }
    }
    
    /**
     * Mettre à jour le badge de notifications
     */
    updateNotificationBadge(count) {
        const badge = document.querySelector('.notification-badge');
        if (badge) {
            const currentCount = parseInt(badge.textContent) || 0;
            const newCount = currentCount + count;
            badge.textContent = newCount;
            badge.style.display = newCount > 0 ? 'flex' : 'none';
            
            // Animation du badge
            badge.style.animation = 'none';
            setTimeout(() => {
                badge.style.animation = 'bounce 0.6s ease-out';
            }, 10);
        }
    }
    
    /**
     * Réinitialiser le badge
     */
    resetNotificationBadge() {
        const badge = document.querySelector('.notification-badge');
        if (badge) {
            badge.textContent = '0';
            badge.style.display = 'none';
        }
    }
    
    /**
     * Changer la fréquence de polling
     */
    setPollingFrequency(seconds) {
        this.pollingFrequency = seconds * 1000;
        console.log(`⏱️ Fréquence polling changée: ${seconds}s`);
        
        if (this.isActive) {
            this.stop();
            this.start();
        }
    }
    
    /**
     * Réinitialiser l'ID de référence (pour debug)
     */
    async resetLastPeseeId() {
        console.log('🔄 Reset de l\'ID de référence...');
        const oldId = this.lastPeseeId;
        this.lastPeseeId = null;
        await this.initializeLastPeseeId();
        console.log(`🔄 Reset terminé: ${oldId} → ${this.lastPeseeId}`);
        return this.lastPeseeId;
    }
    
    /**
     * Obtenir le statut de surveillance
     */
    getStatus() {
        return {
            isActive: this.isActive,
            pollingFrequency: this.pollingFrequency / 1000,
            lastCheckTime: this.lastCheckTime,
            lastPeseeId: this.lastPeseeId
        };
    }
}

// Instance globale
const notificationWatcher = new NotificationWatcher();

// Auto-démarrage quand l'app est prête
document.addEventListener('DOMContentLoaded', () => {
    // Attendre 5 secondes pour laisser l'app se charger complètement
    setTimeout(() => {
        if (document.querySelector('.main-content')) {
            notificationWatcher.start();
            console.log('🚀 Surveillance automatique des pesées démarrée');
        }
    }, 5000);
});

// Export pour utilisation dans d'autres scripts
window.notificationWatcher = notificationWatcher;

console.log('📋 NotificationWatcher v2.0 chargé (avec debug détaillé)');

// Debug : Afficher l'état dans la console toutes les 10 secondes
setInterval(() => {
    if (window.notificationWatcher) {
        const status = notificationWatcher.getStatus();
        console.log('🔍 État surveillance:', {
            actif: status.isActive,
            fréquence: status.pollingFrequency + 's',
            dernièreVérif: status.lastCheckTime?.toLocaleTimeString() || 'Jamais',
            dernierID: status.lastPeseeId
        });
    }
}, 10000);