<?php
/**
 * EnvLoader - Classe pour charger les variables d'environnement
 * Sécurise la configuration en externalisant les credentials
 */
class EnvLoader {
    
    private static $loaded = false;
    private static $variables = [];
    
    /**
     * Charger les variables depuis le fichier .env
     */
    public static function load($envFile = '.env') {
        if (self::$loaded) {
            return;
        }
        
        $envPath = __DIR__ . '/' . $envFile;
        
        if (!file_exists($envPath)) {
            throw new Exception("Fichier d'environnement introuvable: $envPath");
        }
        
        $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
            // Ignorer les commentaires
            if (strpos(trim($line), '#') === 0) {
                continue;
            }
            
            // Parser les variables KEY=VALUE
            if (strpos($line, '=') !== false) {
                list($key, $value) = explode('=', $line, 2);
                $key = trim($key);
                $value = trim($value);
                
                // Supprimer les guillemets si présents
                if (preg_match('/^["\'](.*)["\']/s', $value, $matches)) {
                    $value = $matches[1];
                }
                
                // Stocker dans $_ENV, $_SERVER et notre cache
                $_ENV[$key] = $value;
                $_SERVER[$key] = $value;
                self::$variables[$key] = $value;
            }
        }
        
        self::$loaded = true;
    }
    
    /**
     * Obtenir une variable d'environnement avec valeur par défaut
     */
    public static function get($key, $default = null) {
        // Charger automatiquement si pas encore fait
        if (!self::$loaded) {
            try {
                self::load();
            } catch (Exception $e) {
                // Si le fichier .env n'existe pas, continuer avec les valeurs par défaut
            }
        }
        
        // Priorité: variable d'environnement système, puis .env, puis défaut
        if (isset($_ENV[$key])) {
            return $_ENV[$key];
        }
        
        if (isset(self::$variables[$key])) {
            return self::$variables[$key];
        }
        
        return $default;
    }
    
    /**
     * Vérifier si toutes les variables requises sont définies
     */
    public static function checkRequired($requiredVars) {
        $missing = [];
        
        foreach ($requiredVars as $var) {
            if (self::get($var) === null) {
                $missing[] = $var;
            }
        }
        
        if (!empty($missing)) {
            throw new Exception("Variables d'environnement manquantes: " . implode(', ', $missing));
        }
    }
    
    /**
     * Obtenir toutes les variables (pour debug - attention à la sécurité)
     */
    public static function getAll($hideSensitive = true) {
        $vars = array_merge(self::$variables, $_ENV);
        
        if ($hideSensitive) {
            $sensitiveKeys = ['DB_PASS', 'JWT_SECRET', 'PASSWORD', 'SECRET', 'KEY', 'TOKEN'];
            
            foreach ($vars as $key => $value) {
                foreach ($sensitiveKeys as $sensitive) {
                    if (stripos($key, $sensitive) !== false) {
                        $vars[$key] = '***hidden***';
                        break;
                    }
                }
            }
        }
        
        return $vars;
    }
}
?>