<?php
// tests/test_suite.php - Suite de tests automatisés

error_reporting(E_ALL);
ini_set('display_errors', 1);

class GestionFinancementTestSuite {
    private $baseUrl;
    private $authToken;
    private $testResults = [];
    private $totalTests = 0;
    private $passedTests = 0;
    
    public function __construct($baseUrl) {
        $this->baseUrl = rtrim($baseUrl, '/');
        echo "🧪 SUITE DE TESTS - GESTION FINANCEMENT\n";
        echo "=====================================\n\n";
    }
    
    public function runAllTests() {
        $this->testDatabaseConnection();
        $this->testAuthentication();
        $this->testSitesAPI();
        $this->testProduitsAPI();
        $this->testFournisseursAPI();
        $this->testClientsAPI();
        $this->testAchatsAPI();
        $this->testVentesAPI();
        $this->testReportingAPI();
        $this->testPWAFeatures();
        
        $this->displayResults();
    }
    
    private function test($testName, $callback) {
        $this->totalTests++;
        echo "Testing: $testName... ";
        
        try {
            $result = $callback();
            if ($result) {
                echo "✅ PASS\n";
                $this->passedTests++;
                $this->testResults[] = ['test' => $testName, 'status' => 'PASS', 'error' => null];
            } else {
                echo "❌ FAIL\n";
                $this->testResults[] = ['test' => $testName, 'status' => 'FAIL', 'error' => 'Test returned false'];
            }
        } catch (Exception $e) {
            echo "❌ ERROR: " . $e->getMessage() . "\n";
            $this->testResults[] = ['test' => $testName, 'status' => 'ERROR', 'error' => $e->getMessage()];
        }
    }
    
    private function makeRequest($endpoint, $method = 'GET', $data = null, $requireAuth = true) {
        $url = $this->baseUrl . $endpoint;
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => false, // Pour les tests seulement
        ]);
        
        $headers = ['Content-Type: application/json'];
        
        if ($requireAuth && $this->authToken) {
            $headers[] = 'Authorization: Bearer ' . $this->authToken;
        }
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($data && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: $error");
        }
        
        return [
            'code' => $httpCode,
            'body' => $response,
            'data' => json_decode($response, true)
        ];
    }
    
    private function testDatabaseConnection() {
        $this->test('Database Connection', function() {
            // Test via l'API de santé (à créer)
            $response = $this->makeRequest('/api/health', 'GET', null, false);
            return $response['code'] === 200;
        });
    }
    
    private function testAuthentication() {
        $this->test('Authentication - Login', function() {
            $response = $this->makeRequest('/api/auth/login', 'POST', [
                'email' => 'admin@gestionfinance.com',
                'mot_de_passe' => 'admin123'
            ], false);
            
            if ($response['code'] === 200 && isset($response['data']['data']['token'])) {
                $this->authToken = $response['data']['data']['token'];
                return true;
            }
            return false;
        });
        
        $this->test('Authentication - Verify Token', function() {
            $response = $this->makeRequest('/api/auth/verify', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Authentication - Get Current User', function() {
            $response = $this->makeRequest('/api/auth/me', 'GET');
            return $response['code'] === 200 && isset($response['data']['data']['email']);
        });
    }
    
    private function testSitesAPI() {
        $siteId = null;
        
        $this->test('Sites - List', function() {
            $response = $this->makeRequest('/api/sites', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Sites - Create', function() use (&$siteId) {
            $response = $this->makeRequest('/api/sites', 'POST', [
                'nom' => 'Site Test Automatique',
                'type_site' => 'autre',
                'adresse' => 'Adresse de test',
                'budget_initial' => 1000000
            ]);
            
            if ($response['code'] === 201) {
                $siteId = $response['data']['data']['id'];
                return true;
            }
            return false;
        });
        
        $this->test('Sites - Get Single', function() use ($siteId) {
            if (!$siteId) return false;
            $response = $this->makeRequest("/api/sites/$siteId", 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Sites - Update', function() use ($siteId) {
            if (!$siteId) return false;
            $response = $this->makeRequest("/api/sites/$siteId", 'PUT', [
                'nom' => 'Site Test Modifié'
            ]);
            return $response['code'] === 200;
        });
    }
    
    private function testProduitsAPI() {
        $produitId = null;
        
        $this->test('Produits - List', function() {
            $response = $this->makeRequest('/api/produits', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Produits - Create', function() use (&$produitId) {
            $response = $this->makeRequest('/api/produits', 'POST', [
                'nom' => 'Produit Test',
                'unite_mesure' => 'kg',
                'categorie' => 'autre',
                'description' => 'Produit créé automatiquement pour les tests'
            ]);
            
            if ($response['code'] === 201) {
                $produitId = $response['data']['data']['id'];
                return true;
            }
            return false;
        });
    }
    
    private function testFournisseursAPI() {
        $this->test('Fournisseurs - List', function() {
            $response = $this->makeRequest('/api/fournisseurs', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Fournisseurs - Create', function() {
            $response = $this->makeRequest('/api/fournisseurs', 'POST', [
                'nom' => 'Fournisseur Test',
                'telephone' => '+225 07 12 34 56 78',
                'type_produit' => 'Test'
            ]);
            return $response['code'] === 201;
        });
    }
    
    private function testClientsAPI() {
        $this->test('Clients - List', function() {
            $response = $this->makeRequest('/api/clients', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Clients - Create', function() {
            $response = $this->makeRequest('/api/clients', 'POST', [
                'nom' => 'Client Test',
                'telephone' => '+225 01 23 45 67 89',
                'type_client' => 'particulier'
            ]);
            return $response['code'] === 201;
        });
    }
    
    private function testAchatsAPI() {
        $this->test('Achats - List', function() {
            $response = $this->makeRequest('/api/achats', 'GET');
            return $response['code'] === 200;
        });
        
        // Test de création d'achat (nécessite des données existantes)
        $this->test('Achats - Create (with existing data)', function() {
            // Récupérer les IDs nécessaires
            $sites = $this->makeRequest('/api/sites', 'GET');
            $produits = $this->makeRequest('/api/produits', 'GET');
            $fournisseurs = $this->makeRequest('/api/fournisseurs', 'GET');
            
            if ($sites['code'] === 200 && $produits['code'] === 200 && $fournisseurs['code'] === 200) {
                $sitesData = $sites['data']['data'];
                $produitsData = $produits['data']['data'];
                $fournisseursData = $fournisseurs['data']['data'];
                
                if (!empty($sitesData) && !empty($produitsData) && !empty($fournisseursData)) {
                    $response = $this->makeRequest('/api/achats', 'POST', [
                        'site_id' => $sitesData[0]['id'],
                        'fournisseur_id' => $fournisseursData[0]['id'],
                        'produit_id' => $produitsData[0]['id'],
                        'quantite' => 100,
                        'prix_unitaire' => 500,
                        'date_achat' => date('Y-m-d')
                    ]);
                    return $response['code'] === 201;
                }
            }
            return false;
        });
    }
    
    private function testVentesAPI() {
        $this->test('Ventes - List', function() {
            $response = $this->makeRequest('/api/ventes', 'GET');
            return $response['code'] === 200;
        });
    }
    
    private function testReportingAPI() {
        $this->test('Reporting - Dashboard', function() {
            $response = $this->makeRequest('/api/reporting/dashboard', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Reporting - Stocks', function() {
            $response = $this->makeRequest('/api/reporting/stocks', 'GET');
            return $response['code'] === 200;
        });
        
        $this->test('Reporting - Rentabilité', function() {
            $response = $this->makeRequest('/api/reporting/rentabilite', 'GET');
            return $response['code'] === 200;
        });
    }
    
    private function testPWAFeatures() {
        $this->test('PWA - Manifest', function() {
            $response = $this->makeRequest('/manifest.json', 'GET', null, false);
            return $response['code'] === 200 && isset($response['data']['name']);
        });
        
        $this->test('PWA - Service Worker', function() {
            $response = $this->makeRequest('/sw.js', 'GET', null, false);
            return $response['code'] === 200;
        });
        
        $this->test('PWA - Icons', function() {
            $response = $this->makeRequest('/icons/icon-192x192.png', 'GET', null, false);
            return $response['code'] === 200;
        });
    }
    
    private function displayResults() {
        echo "\n" . str_repeat("=", 50) . "\n";
        echo "RÉSULTATS DES TESTS\n";
        echo str_repeat("=", 50) . "\n\n";
        
        echo "Total des tests: {$this->totalTests}\n";
        echo "Tests réussis: {$this->passedTests}\n";
        echo "Tests échoués: " . ($this->totalTests - $this->passedTests) . "\n";
        echo "Taux de réussite: " . round(($this->passedTests / $this->totalTests) * 100, 2) . "%\n\n";
        
        // Afficher les tests échoués
        $failedTests = array_filter($this->testResults, function($test) {
            return $test['status'] !== 'PASS';
        });
        
        if (!empty($failedTests)) {
            echo "TESTS ÉCHOUÉS:\n";
            echo str_repeat("-", 30) . "\n";
            foreach ($failedTests as $test) {
                echo "❌ {$test['test']}: {$test['error']}\n";
            }
            echo "\n";
        }
        
        if ($this->passedTests === $this->totalTests) {
            echo "🎉 TOUS LES TESTS SONT PASSÉS! L'application est prête.\n";
        } else {
            echo "⚠️  Certains tests ont échoué. Vérifiez la configuration.\n";
        }
    }
}

// Script de maintenance
class MaintenanceManager {
    private $appDir;
    private $logFile;
    
    public function __construct($appDir) {
        $this->appDir = $appDir;
        $this->logFile = $appDir . '/logs/maintenance.log';
    }
    
    public function runMaintenance() {
        $this->log("Début de la maintenance automatique");
        
        $this->cleanupOldLogs();
        $this->optimizeDatabase();
        $this->clearCache();
        $this->checkDiskSpace();
        $this->validateConfiguration();
        $this->updateStatistics();
        
        $this->log("Maintenance terminée avec succès");
    }
    
    private function cleanupOldLogs() {
        $this->log("Nettoyage des anciens logs");
        
        $logDir = $this->appDir . '/logs';
        $files = glob($logDir . '/*.log');
        
        foreach ($files as $file) {
            if (filemtime($file) < strtotime('-30 days')) {
                unlink($file);
                $this->log("Suppression du fichier log: " . basename($file));
            }
        }
    }
    
    private function optimizeDatabase() {
        $this->log("Optimisation de la base de données");
        
        try {
            require_once $this->appDir . '/api/config.php';
            $db = Database::getInstance()->getConnection();
            
            $tables = ['sites', 'budgets', 'achats', 'ventes', 'stocks', 'fournisseurs', 'clients', 'produits', 'transactions'];
            
            foreach ($tables as $table) {
                $stmt = $db->prepare("OPTIMIZE TABLE $table");
                $stmt->execute();
            }
            
            $this->log("Base de données optimisée");
        } catch (Exception $e) {
            $this->log("Erreur optimisation DB: " . $e->getMessage());
        }
    }
    
    private function clearCache() {
        $this->log("Nettoyage du cache");
        
        $cacheDir = $this->appDir . '/cache';
        if (is_dir($cacheDir)) {
            $files = glob($cacheDir . '/*');
            foreach ($files as $file) {
                if (is_file($file) && filemtime($file) < strtotime('-1 day')) {
                    unlink($file);
                }
            }
        }
    }
    
    private function checkDiskSpace() {
        $diskUsage = disk_free_space($this->appDir) / disk_total_space($this->appDir) * 100;
        
        if ($diskUsage < 10) {
            $this->log("ALERTE: Espace disque faible (" . round($diskUsage, 2) . "% libre)");
        } else {
            $this->log("Espace disque OK (" . round($diskUsage, 2) . "% libre)");
        }
    }
    
    private function validateConfiguration() {
        $this->log("Validation de la configuration");
        
        $envFile = $this->appDir . '/.env';
        if (!file_exists($envFile)) {
            $this->log("ERREUR: Fichier .env manquant");
            return;
        }
        
        $requiredVars = ['DB_HOST', 'DB_NAME', 'DB_USER', 'DB_PASS', 'JWT_SECRET'];
        $env = parse_ini_file($envFile);
        
        foreach ($requiredVars as $var) {
            if (!isset($env[$var]) || empty($env[$var])) {
                $this->log("ERREUR: Variable d'environnement manquante: $var");
            }
        }
    }
    
    private function updateStatistics() {
        $this->log("Mise à jour des statistiques");
        
        try {
            require_once $this->appDir . '/api/config.php';
            $db = Database::getInstance()->getConnection();
            
            // Mettre à jour les statistiques des sites
            $stmt = $db->prepare("
                INSERT INTO site_statistics (site_id, date_stats, total_ventes, total_achats, marge_brute)
                SELECT s.id, CURDATE(), 
                       COALESCE(SUM(v.montant_total), 0),
                       COALESCE(SUM(a.cout_total), 0),
                       COALESCE(SUM(v.marge_brute), 0)
                FROM sites s
                LEFT JOIN ventes v ON s.id = v.site_id AND DATE(v.date_vente) = CURDATE()
                LEFT JOIN achats a ON s.id = a.site_id AND DATE(a.date_achat) = CURDATE()
                GROUP BY s.id
                ON DUPLICATE KEY UPDATE
                total_ventes = VALUES(total_ventes),
                total_achats = VALUES(total_achats),
                marge_brute = VALUES(marge_brute)
            ");
            $stmt->execute();
            
            $this->log("Statistiques mises à jour");
        } catch (Exception $e) {
            $this->log("Erreur mise à jour statistiques: " . $e->getMessage());
        }
    }
    
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
        echo $logMessage;
    }
}

// Point d'entrée pour les tests
if (php_sapi_name() === 'cli') {
    if ($argc < 2) {
        echo "Usage: php test_suite.php <base_url> [maintenance]\n";
        echo "Exemple: php test_suite.php http://localhost\n";
        echo "Pour la maintenance: php test_suite.php /var/www/gestion-financement maintenance\n";
        exit(1);
    }
    
    if (isset($argv[2]) && $argv[2] === 'maintenance') {
        $maintenance = new MaintenanceManager($argv[1]);
        $maintenance->runMaintenance();
    } else {
        $testSuite = new GestionFinancementTestSuite($argv[1]);
        $testSuite->runAllTests();
    }
}

?>

<!-- Script de diagnostic HTML -->
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnostic - Gestion Financement</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .test { margin: 10px 0; padding: 10px; border-radius: 5px; }
        .pass { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; }
        .fail { background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; }
        .warning { background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; }
        button { padding: 10px 20px; margin: 5px; cursor: pointer; }
        #results { margin-top: 20px; }
        .hidden { display: none; }
    </style>
</head>
<body>
    <h1>🔧 Diagnostic Gestion Financement</h1>
    
    <div>
        <button onclick="runDiagnostic()">Lancer le diagnostic</button>
        <button onclick="testPWA()">Tester PWA</button>
        <button onclick="testAPI()">Tester API</button>
        <button onclick="clearResults()">Effacer</button>
    </div>
    
    <div id="results"></div>
    
    <script>
        function addResult(test, status, message) {
            const results = document.getElementById('results');
            const div = document.createElement('div');
            div.className = `test ${status}`;
            
            const icon = status === 'pass' ? '✅' : status === 'fail' ? '❌' : '⚠️';
            div.innerHTML = `${icon} <strong>${test}:</strong> ${message}`;
            
            results.appendChild(div);
        }
        
        function clearResults() {
            document.getElementById('results').innerHTML = '';
        }
        
        async function runDiagnostic() {
            clearResults();
            addResult('Diagnostic', 'warning', 'Démarrage du diagnostic...');
            
            // Test de base
            await testBasicFeatures();
            await testLocalStorage();
            await testIndexedDB();
            await testServiceWorker();
            await testNetworkStatus();
        }
        
        async function testBasicFeatures() {
            // Test JavaScript
            try {
                const test = () => true;
                if (test()) {
                    addResult('JavaScript', 'pass', 'JavaScript fonctionne correctement');
                }
            } catch (e) {
                addResult('JavaScript', 'fail', 'Erreur JavaScript: ' + e.message);
            }
            
            // Test LocalStorage
            try {
                localStorage.setItem('test', 'value');
                localStorage.removeItem('test');
                addResult('LocalStorage', 'pass', 'LocalStorage disponible');
            } catch (e) {
                addResult('LocalStorage', 'fail', 'LocalStorage non disponible');
            }
            
            // Test SessionStorage
            try {
                sessionStorage.setItem('test', 'value');
                sessionStorage.removeItem('test');
                addResult('SessionStorage', 'pass', 'SessionStorage disponible');
            } catch (e) {
                addResult('SessionStorage', 'fail', 'SessionStorage non disponible');
            }
        }
        
        async function testLocalStorage() {
            try {
                const testKey = 'diagnostic_test';
                const testValue = 'test_data_' + Date.now();
                
                localStorage.setItem(testKey, testValue);
                const retrieved = localStorage.getItem(testKey);
                localStorage.removeItem(testKey);
                
                if (retrieved === testValue) {
                    addResult('LocalStorage R/W', 'pass', 'Lecture/écriture LocalStorage OK');
                } else {
                    addResult('LocalStorage R/W', 'fail', 'Problème lecture/écriture LocalStorage');
                }
            } catch (e) {
                addResult('LocalStorage R/W', 'fail', 'Erreur: ' + e.message);
            }
        }
        
        async function testIndexedDB() {
            if (!window.indexedDB) {
                addResult('IndexedDB', 'fail', 'IndexedDB non supporté');
                return;
            }
            
            try {
                const request = indexedDB.open('test_db', 1);
                
                request.onsuccess = () => {
                    addResult('IndexedDB', 'pass', 'IndexedDB disponible et fonctionnel');
                    request.result.close();
                    indexedDB.deleteDatabase('test_db');
                };
                
                request.onerror = () => {
                    addResult('IndexedDB', 'fail', 'Erreur d\'ouverture IndexedDB');
                };
                
                request.onupgradeneeded = (event) => {
                    const db = event.target.result;
                    db.createObjectStore('test_store', { keyPath: 'id' });
                };
            } catch (e) {
                addResult('IndexedDB', 'fail', 'Erreur IndexedDB: ' + e.message);
            }
        }
        
        async function testServiceWorker() {
            if (!('serviceWorker' in navigator)) {
                addResult('Service Worker', 'fail', 'Service Worker non supporté');
                return;
            }
            
            try {
                const registration = await navigator.serviceWorker.getRegistration();
                if (registration) {
                    addResult('Service Worker', 'pass', 'Service Worker enregistré et actif');
                } else {
                    addResult('Service Worker', 'warning', 'Service Worker non enregistré');
                }
            } catch (e) {
                addResult('Service Worker', 'fail', 'Erreur Service Worker: ' + e.message);
            }
        }
        
        async function testNetworkStatus() {
            const online = navigator.onLine;
            addResult('Connexion réseau', online ? 'pass' : 'warning', 
                     online ? 'En ligne' : 'Hors ligne');
            
            // Test de ping vers l'API
            try {
                const response = await fetch('/api/health', { 
                    method: 'GET',
                    cache: 'no-cache'
                });
                
                if (response.ok) {
                    addResult('API Health Check', 'pass', 'API accessible');
                } else {
                    addResult('API Health Check', 'warning', `API répond avec le code ${response.status}`);
                }
            } catch (e) {
                addResult('API Health Check', 'fail', 'API non accessible: ' + e.message);
            }
        }
        
        async function testPWA() {
            clearResults();
            addResult('Test PWA', 'warning', 'Test des fonctionnalités PWA...');
            
            // Test Manifest
            try {
                const response = await fetch('/manifest.json');
                if (response.ok) {
                    const manifest = await response.json();
                    addResult('PWA Manifest', 'pass', `Manifest OK: ${manifest.name}`);
                } else {
                    addResult('PWA Manifest', 'fail', 'Manifest non trouvé');
                }
            } catch (e) {
                addResult('PWA Manifest', 'fail', 'Erreur manifest: ' + e.message);
            }
            
            // Test d'installation
            if ('beforeinstallprompt' in window) {
                addResult('PWA Installation', 'pass', 'Installation PWA supportée');
            } else {
                addResult('PWA Installation', 'warning', 'Installation PWA non supportée ou déjà installée');
            }
            
            // Test des notifications
            if ('Notification' in window) {
                const permission = Notification.permission;
                addResult('PWA Notifications', permission === 'granted' ? 'pass' : 'warning', 
                         `Notifications: ${permission}`);
            } else {
                addResult('PWA Notifications', 'fail', 'Notifications non supportées');
            }
        }
        
        async function testAPI() {
            clearResults();
            addResult('Test API', 'warning', 'Test des endpoints API...');
            
            const endpoints = [
                '/api/sites',
                '/api/produits', 
                '/api/fournisseurs',
                '/api/clients',
                '/api/achats',
                '/api/ventes'
            ];
            
            for (const endpoint of endpoints) {
                try {
                    const response = await fetch(endpoint, {
                        headers: {
                            'Authorization': 'Bearer ' + (localStorage.getItem('authToken') || 'test')
                        }
                    });
                    
                    const status = response.status === 401 ? 'warning' : (response.ok ? 'pass' : 'fail');
                    const message = response.status === 401 ? 'Non authentifié (normal)' : 
                                   response.ok ? 'OK' : `Erreur ${response.status}`;
                    
                    addResult(`API ${endpoint}`, status, message);
                } catch (e) {
                    addResult(`API ${endpoint}`, 'fail', 'Erreur: ' + e.message);
                }
            }
        }
        
        // Auto-run diagnostic on page load
        document.addEventListener('DOMContentLoaded', function() {
            if (window.location.search.includes('autorun')) {
                runDiagnostic();
            }
        });
    </script>
</body>
</html>