// sw.js - Service Worker pour PWA Gestion Financement
const CACHE_NAME = 'gestion-finance-v1.0.0';
const STATIC_CACHE = 'static-v1.0.0';
const DYNAMIC_CACHE = 'dynamic-v1.0.0';

// Fichiers à mettre en cache (cache statique)
const STATIC_FILES = [
    '/',
    '/index.html',
    '/manifest.json',
    '/styles/main.css',
    '/styles/components.css',
    '/js/database.js',
    '/js/sync.js',
    '/js/app.js',
    '/js/dashboard.js',
    '/js/sites.js',
    '/icons/icon-192x192.png',
    '/icons/icon-512x512.png',
    // Ressources externes essentielles
    'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap',
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css',
    'https://cdnjs.cloudflare.com/ajax/libs/chart.js/3.9.1/chart.min.js'
];

// URLs à exclure du cache
const EXCLUDED_URLS = [
    '/api/',
    '/sync.php',
    '/upload.php'
];

// Installation du Service Worker
self.addEventListener('install', event => {
    console.log('Service Worker: Installing...');
    
    event.waitUntil(
        caches.open(STATIC_CACHE)
            .then(cache => {
                console.log('Service Worker: Caching static files');
                return cache.addAll(STATIC_FILES);
            })
            .then(() => {
                console.log('Service Worker: Static files cached');
                return self.skipWaiting();
            })
            .catch(error => {
                console.error('Service Worker: Error caching static files', error);
            })
    );
});

// Activation du Service Worker
self.addEventListener('activate', event => {
    console.log('Service Worker: Activating...');
    
    event.waitUntil(
        caches.keys()
            .then(cacheNames => {
                return Promise.all(
                    cacheNames.map(cacheName => {
                        // Supprimer les anciens caches
                        if (cacheName !== STATIC_CACHE && cacheName !== DYNAMIC_CACHE) {
                            console.log('Service Worker: Removing old cache', cacheName);
                            return caches.delete(cacheName);
                        }
                    })
                );
            })
            .then(() => {
                console.log('Service Worker: Activated');
                return self.clients.claim();
            })
    );
});

// Interception des requêtes (stratégie de cache)
self.addEventListener('fetch', event => {
    const { request } = event;
    const url = new URL(request.url);
    
    // Ignorer les requêtes non-HTTP
    if (!request.url.startsWith('http')) {
        return;
    }
    
    // Stratégie pour les fichiers statiques
    if (STATIC_FILES.includes(request.url) || isStaticAsset(request.url)) {
        event.respondWith(cacheFirstStrategy(request));
        return;
    }
    
    // Stratégie pour l'API (Network First avec fallback)
    if (isApiRequest(request.url)) {
        event.respondWith(networkFirstStrategy(request));
        return;
    }
    
    // Stratégie pour les images et autres assets
    if (isImageRequest(request.url)) {
        event.respondWith(cacheFirstStrategy(request));
        return;
    }
    
    // Stratégie par défaut pour les pages HTML
    event.respondWith(networkFirstStrategy(request));
});

// Stratégie Cache First (cache en priorité)
async function cacheFirstStrategy(request) {
    try {
        const cachedResponse = await caches.match(request);
        if (cachedResponse) {
            return cachedResponse;
        }
        
        const networkResponse = await fetch(request);
        
        // Mettre en cache si la réponse est valide
        if (networkResponse && networkResponse.status === 200) {
            const cache = await caches.open(DYNAMIC_CACHE);
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.error('Cache First Strategy failed:', error);
        
        // Fallback pour les pages HTML
        if (request.destination === 'document') {
            return caches.match('/index.html');
        }
        
        // Fallback pour les images
        if (isImageRequest(request.url)) {
            return caches.match('/icons/offline-image.png');
        }
        
        throw error;
    }
}

// Stratégie Network First (réseau en priorité)
async function networkFirstStrategy(request) {
    try {
        const networkResponse = await fetch(request);
        
        // Mettre en cache si c'est une réponse valide et pas une requête API exclue
        if (networkResponse && networkResponse.status === 200 && !isExcludedFromCache(request.url)) {
            const cache = await caches.open(DYNAMIC_CACHE);
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.log('Network failed, trying cache:', error);
        
        const cachedResponse = await caches.match(request);
        if (cachedResponse) {
            return cachedResponse;
        }
        
        // Fallback pour les pages HTML en mode offline
        if (request.destination === 'document') {
            return caches.match('/index.html');
        }
        
        // Réponse offline pour les requêtes API
        if (isApiRequest(request.url)) {
            return new Response(
                JSON.stringify({
                    error: 'Offline',
                    message: 'Cette fonctionnalité nécessite une connexion internet'
                }),
                {
                    status: 503,
                    statusText: 'Service Unavailable',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                }
            );
        }
        
        throw error;
    }
}

// Fonctions utilitaires
function isStaticAsset(url) {
    return url.includes('.css') || 
           url.includes('.js') || 
           url.includes('.woff') || 
           url.includes('.woff2') ||
           url.includes('/fonts/');
}

function isApiRequest(url) {
    return url.includes('/api/') || 
           url.includes('.php') ||
           url.includes('/sync/');
}

function isImageRequest(url) {
    return url.includes('.png') || 
           url.includes('.jpg') || 
           url.includes('.jpeg') || 
           url.includes('.gif') || 
           url.includes('.svg') ||
           url.includes('.webp');
}

function isExcludedFromCache(url) {
    return EXCLUDED_URLS.some(excludedUrl => url.includes(excludedUrl));
}

// Gestion des messages depuis l'application
self.addEventListener('message', event => {
    if (event.data && event.data.type) {
        switch (event.data.type) {
            case 'SKIP_WAITING':
                self.skipWaiting();
                break;
                
            case 'GET_VERSION':
                event.ports[0].postMessage({ version: CACHE_NAME });
                break;
                
            case 'CACHE_URLS':
                if (event.data.urls) {
                    cacheUrls(event.data.urls);
                }
                break;
                
            case 'CLEAR_CACHE':
                clearCache();
                break;
        }
    }
});

// Fonction pour mettre en cache des URLs spécifiques
async function cacheUrls(urls) {
    try {
        const cache = await caches.open(DYNAMIC_CACHE);
        await cache.addAll(urls);
        console.log('URLs cached successfully:', urls);
    } catch (error) {
        console.error('Error caching URLs:', error);
    }
}

// Fonction pour vider le cache
async function clearCache() {
    try {
        const cacheNames = await caches.keys();
        await Promise.all(
            cacheNames.map(cacheName => caches.delete(cacheName))
        );
        console.log('All caches cleared');
    } catch (error) {
        console.error('Error clearing cache:', error);
    }
}

// Synchronisation en arrière-plan
self.addEventListener('sync', event => {
    if (event.tag === 'background-sync') {
        event.waitUntil(doBackgroundSync());
    }
});

async function doBackgroundSync() {
    try {
        // Envoyer les données en attente de synchronisation
        const response = await fetch('/api/sync-background.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'sync_pending_data'
            })
        });
        
        if (response.ok) {
            console.log('Background sync completed successfully');
            
            // Notifier l'application principale
            const clients = await self.clients.matchAll();
            clients.forEach(client => {
                client.postMessage({
                    type: 'SYNC_COMPLETED',
                    success: true
                });
            });
        }
    } catch (error) {
        console.error('Background sync failed:', error);
    }
}

// Notifications push (pour futures fonctionnalités)
self.addEventListener('push', event => {
    if (event.data) {
        const data = event.data.json();
        
        const options = {
            body: data.body || 'Nouvelle notification',
            icon: '/icons/icon-192x192.png',
            badge: '/icons/badge-72x72.png',
            vibrate: [100, 50, 100],
            data: data.data || {},
            actions: [
                {
                    action: 'view',
                    title: 'Voir',
                    icon: '/icons/view-action.png'
                },
                {
                    action: 'dismiss',
                    title: 'Ignorer',
                    icon: '/icons/dismiss-action.png'
                }
            ]
        };
        
        event.waitUntil(
            self.registration.showNotification(data.title || 'Gestion Finance', options)
        );
    }
});

// Gestion des clics sur les notifications
self.addEventListener('notificationclick', event => {
    event.notification.close();
    
    if (event.action === 'view') {
        // Ouvrir l'application
        event.waitUntil(
            clients.openWindow('/')
        );
    }
    // Pour 'dismiss' ou clic général, on ferme juste la notification
});

// Gestion des erreurs globales
self.addEventListener('error', event => {
    console.error('Service Worker error:', event.error);
});

self.addEventListener('unhandledrejection', event => {
    console.error('Service Worker unhandled promise rejection:', event.reason);
});

console.log('Service Worker loaded successfully');