// js/stocks.js - Gestion des stocks

// Variables globales pour les stocks
let currentStocks = [];
let stocksChart = null;

// Chargement des données des stocks
async function loadStocksData() {
    try {
        showStocksLoading(true);
        
        // Charger les stocks
        await loadStocks();
        
        // Afficher les stocks
        displayStocksGrid(currentStocks);
        
        // Charger les graphiques
        await loadStocksChart();
        
        // Charger les alertes
        await loadStocksAlertes();
        
        showStocksLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement stocks:', error);
        showError('Erreur lors du chargement des stocks');
        showStocksLoading(false);
    }
}

// Chargement de la liste des stocks
async function loadStocks() {
    try {
        // Charger depuis l'API ou la base locale
        if (isOnline) {
            const response = await fetch('/api/reporting/stocks', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                currentStocks = data.data.stocks_detailles || [];
            } else {
                throw new Error('Erreur API');
            }
        } else {
            // Fallback vers la base locale
            currentStocks = await loadStocksFromLocal();
        }
        
    } catch (error) {
        console.error('Erreur chargement stocks:', error);
        currentStocks = await loadStocksFromLocal();
    }
}

async function loadStocksFromLocal() {
    try {
        const stocks = await dbManager.readAll('stocks');
        const sites = await dbManager.readAll('sites');
        const produits = await dbManager.readAll('produits');
        
        return stocks.filter(s => s.quantite_actuelle > 0).map(stock => {
            const site = sites.find(s => s.id === stock.site_id);
            const produit = produits.find(p => p.id === stock.produit_id);
            
            return {
                ...stock,
                site_nom: site ? site.nom : 'Site inconnu',
                produit_nom: produit ? produit.nom : 'Produit inconnu',
                unite_mesure: produit ? produit.unite_mesure : 'kg'
            };
        });
    } catch (error) {
        console.error('Erreur chargement stocks local:', error);
        return [];
    }
}

// Affichage de la grille des stocks
function displayStocksGrid(stocks) {
    const stocksContainer = document.getElementById('stocks-container');
    if (!stocksContainer) return;
    
    if (stocks.length === 0) {
        stocksContainer.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-boxes"></i>
                <h3>Aucun stock disponible</h3>
                <p>Les stocks apparaîtront ici après les premiers achats</p>
            </div>
        `;
        return;
    }
    
    // Grouper les stocks par site
    const stocksParSite = groupBy(stocks, 'site_nom');
    
    const html = Object.entries(stocksParSite).map(([siteName, siteStocks]) => `
        <div class="site-stocks-card">
            <div class="site-stocks-header">
                <h3>${siteName}</h3>
                <div class="site-stats">
                    <span class="stat">
                        <i class="fas fa-cubes"></i>
                        ${siteStocks.length} produit(s)
                    </span>
                    <span class="stat">
                        <i class="fas fa-euro-sign"></i>
                        ${formatCurrency(siteStocks.reduce((sum, s) => sum + s.valeur_stock, 0))}
                    </span>
                </div>
            </div>
            <div class="stocks-table">
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Quantité</th>
                            <th>Prix moyen</th>
                            <th>Valeur stock</th>
                            <th>Statut</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${siteStocks.map(stock => createStockRow(stock)).join('')}
                    </tbody>
                </table>
            </div>
        </div>
    `).join('');
    
    stocksContainer.innerHTML = html;
}

// Création d'une ligne de stock
function createStockRow(stock) {
    const alertLevel = getStockAlertLevel(stock.quantite_actuelle);
    
    return `
        <tr class="stock-row ${alertLevel}">
            <td>
                <div class="produit-info">
                    <strong>${stock.produit_nom}</strong>
                    <small>${stock.unite_mesure}</small>
                </div>
            </td>
            <td>
                <span class="quantite">${stock.quantite_actuelle}</span>
                ${stock.unite_mesure}
            </td>
            <td>${formatCurrency(stock.prix_moyen_pondere)}</td>
            <td><strong>${formatCurrency(stock.valeur_stock)}</strong></td>
            <td>
                <span class="badge ${getStockStatusClass(alertLevel)}">
                    ${getStockStatusText(alertLevel)}
                </span>
            </td>
        </tr>
    `;
}

// Déterminer le niveau d'alerte du stock
function getStockAlertLevel(quantite) {
    if (quantite <= 10) return 'critique';
    if (quantite <= 50) return 'faible';
    return 'normal';
}

function getStockStatusClass(level) {
    const classes = {
        'critique': 'badge-danger',
        'faible': 'badge-warning',
        'normal': 'badge-success'
    };
    return classes[level] || 'badge-secondary';
}

function getStockStatusText(level) {
    const texts = {
        'critique': 'Stock critique',
        'faible': 'Stock faible',
        'normal': 'Stock normal'
    };
    return texts[level] || level;
}

// Chargement du graphique des stocks
async function loadStocksChart() {
    const ctx = document.getElementById('stocks-chart');
    if (!ctx) return;
    
    try {
        // Préparer les données pour le graphique
        const stocksParProduit = {};
        
        currentStocks.forEach(stock => {
            if (!stocksParProduit[stock.produit_nom]) {
                stocksParProduit[stock.produit_nom] = {
                    quantite: 0,
                    valeur: 0
                };
            }
            stocksParProduit[stock.produit_nom].quantite += stock.quantite_actuelle;
            stocksParProduit[stock.produit_nom].valeur += stock.valeur_stock;
        });
        
        const labels = Object.keys(stocksParProduit);
        const valeurs = Object.values(stocksParProduit).map(p => p.valeur);
        
        // Détruire le graphique existant
        if (stocksChart) {
            stocksChart.destroy();
        }
        
        // Créer le nouveau graphique
        stocksChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    data: valeurs,
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0',
                        '#9966FF', '#FF9F40', '#FF6384', '#C9CBCF'
                    ],
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed / total) * 100).toFixed(1);
                                return `${context.label}: ${formatCurrency(context.parsed)} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
        
    } catch (error) {
        console.error('Erreur création graphique stocks:', error);
    }
}

// Chargement des alertes de stock
async function loadStocksAlertes() {
    const alertesContainer = document.getElementById('stocks-alertes');
    if (!alertesContainer) return;
    
    const stocksCritiques = currentStocks.filter(s => getStockAlertLevel(s.quantite_actuelle) === 'critique');
    const stocksFaibles = currentStocks.filter(s => getStockAlertLevel(s.quantite_actuelle) === 'faible');
    
    if (stocksCritiques.length === 0 && stocksFaibles.length === 0) {
        alertesContainer.innerHTML = `
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                Tous les stocks sont à un niveau normal
            </div>
        `;
        return;
    }
    
    let alertes = '';
    
    if (stocksCritiques.length > 0) {
        alertes += `
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i>
                <strong>Stocks critiques (${stocksCritiques.length}):</strong>
                ${stocksCritiques.map(s => `${s.produit_nom} (${s.site_nom}): ${s.quantite_actuelle} ${s.unite_mesure}`).join(', ')}
            </div>
        `;
    }
    
    if (stocksFaibles.length > 0) {
        alertes += `
            <div class="alert alert-warning">
                <i class="fas fa-exclamation-circle"></i>
                <strong>Stocks faibles (${stocksFaibles.length}):</strong>
                ${stocksFaibles.map(s => `${s.produit_nom} (${s.site_nom}): ${s.quantite_actuelle} ${s.unite_mesure}`).join(', ')}
            </div>
        `;
    }
    
    alertesContainer.innerHTML = alertes;
}

function showStocksLoading(show) {
    const stocksContainer = document.getElementById('stocks-container');
    if (!stocksContainer) return;
    
    if (show) {
        stocksContainer.innerHTML = `
            <div class="loading-container">
                <div class="loading-spinner"></div>
                <p>Chargement des stocks...</p>
            </div>
        `;
    }
}

// Export des fonctions
window.loadStocksData = loadStocksData;

//////////////////////////////////////////////
// js/reporting.js - Gestion du reporting
//////////////////////////////////////////////

// Variables globales pour le reporting
let reportingCharts = {};
let currentReportData = null;

// Chargement des données de reporting
async function loadReportingData() {
    try {
        showReportingLoading(true);
        
        // Charger les données par défaut du dashboard
        await loadDashboardReporting();
        
        // Initialiser les contrôles
        initReportingControls();
        
        showReportingLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement reporting:', error);
        showError('Erreur lors du chargement des rapports');
        showReportingLoading(false);
    }
}

// Chargement du reporting dashboard
async function loadDashboardReporting() {
    try {
        if (isOnline) {
            const response = await fetch('/api/reporting/dashboard', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                currentReportData = data.data;
                displayDashboardReporting(currentReportData);
            } else {
                throw new Error('Erreur API reporting');
            }
        } else {
            // Générer le reporting depuis les données locales
            currentReportData = await generateLocalReporting();
            displayDashboardReporting(currentReportData);
        }
        
    } catch (error) {
        console.error('Erreur chargement dashboard reporting:', error);
        currentReportData = await generateLocalReporting();
        displayDashboardReporting(currentReportData);
    }
}

// Génération du reporting depuis les données locales
async function generateLocalReporting() {
    try {
        const sites = await dbManager.readAll('sites');
        const ventes = await dbManager.readAll('ventes');
        const achats = await dbManager.readAll('achats');
        
        // KPI globaux
        const currentMonth = new Date().toISOString().substring(0, 7);
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        const ventesMonth = ventes.filter(v => 
            v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
        );
        const achatsMonth = achats.filter(a => 
            a.date_achat >= startOfMonth && a.date_achat <= endOfMonth
        );
        
        const kpi = {
            total_ventes: ventesMonth.reduce((sum, v) => sum + v.montant_total, 0),
            total_achats: achatsMonth.reduce((sum, a) => sum + a.cout_total, 0),
            marge_totale: ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0),
            nb_transactions: ventesMonth.length + achatsMonth.length
        };
        
        // Évolution sur 6 mois
        const evolution = [];
        for (let i = 5; i >= 0; i--) {
            const date = new Date();
            date.setMonth(date.getMonth() - i);
            const mois = date.toISOString().substring(0, 7);
            const startDate = `${mois}-01`;
            const endDate = new Date(date.getFullYear(), date.getMonth() + 1, 0)
                .toISOString().substring(0, 10);
            
            const ventesThisMonth = ventes.filter(v => 
                v.date_vente >= startDate && v.date_vente <= endDate
            );
            const achatsThisMonth = achats.filter(a => 
                a.date_achat >= startDate && a.date_achat <= endDate
            );
            
            evolution.push({
                mois: mois,
                achats: achatsThisMonth.reduce((sum, a) => sum + a.cout_total, 0),
                ventes: ventesThisMonth.reduce((sum, v) => sum + v.montant_total, 0),
                marge: ventesThisMonth.reduce((sum, v) => sum + v.marge_brute, 0)
            });
        }
        
        // Top sites
        const topSites = sites.map(site => {
            const siteVentes = ventesMonth.filter(v => v.site_id === site.id);
            return {
                nom: site.nom,
                ventes_mois: siteVentes.reduce((sum, v) => sum + v.montant_total, 0),
                marge_mois: siteVentes.reduce((sum, v) => sum + v.marge_brute, 0)
            };
        }).sort((a, b) => b.ventes_mois - a.ventes_mois);
        
        return {
            kpi,
            evolution_mensuelle: evolution,
            top_sites: topSites
        };
        
    } catch (error) {
        console.error('Erreur génération reporting local:', error);
        return {
            kpi: { total_ventes: 0, total_achats: 0, marge_totale: 0, nb_transactions: 0 },
            evolution_mensuelle: [],
            top_sites: []
        };
    }
}

// Affichage du dashboard de reporting
function displayDashboardReporting(data) {
    // Afficher les KPI
    displayReportingKPI(data.kpi);
    
    // Créer les graphiques
    createEvolutionChart(data.evolution_mensuelle);
    createTopSitesChart(data.top_sites);
    
    // Afficher les tableaux
    displayTopSitesTable(data.top_sites);
}

// Affichage des KPI de reporting
function displayReportingKPI(kpi) {
    const kpiContainer = document.querySelector('.reporting-kpi');
    if (!kpiContainer) return;
    
    kpiContainer.innerHTML = `
        <div class="kpi-card">
            <div class="kpi-icon" style="background: linear-gradient(135deg, #4facfe, #00f2fe);">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="kpi-content">
                <h3>${formatCurrency(kpi.total_ventes)}</h3>
                <p>Chiffre d'affaires</p>
            </div>
        </div>
        <div class="kpi-card">
            <div class="kpi-icon" style="background: linear-gradient(135deg, #f093fb, #f5576c);">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="kpi-content">
                <h3>${formatCurrency(kpi.total_achats)}</h3>
                <p>Total achats</p>
            </div>
        </div>
        <div class="kpi-card">
            <div class="kpi-icon" style="background: linear-gradient(135deg, #43e97b, #38f9d7);">
                <i class="fas fa-coins"></i>
            </div>
            <div class="kpi-content">
                <h3>${formatCurrency(kpi.marge_totale)}</h3>
                <p>Marge brute</p>
            </div>
        </div>
        <div class="kpi-card">
            <div class="kpi-icon" style="background: linear-gradient(135deg, #667eea, #764ba2);">
                <i class="fas fa-exchange-alt"></i>
            </div>
            <div class="kpi-content">
                <h3>${kpi.nb_transactions}</h3>
                <p>Transactions</p>
            </div>
        </div>
    `;
}

// Création du graphique d'évolution
function createEvolutionChart(evolution) {
    const ctx = document.getElementById('evolution-chart');
    if (!ctx) return;
    
    // Détruire le graphique existant
    if (reportingCharts.evolution) {
        reportingCharts.evolution.destroy();
    }
    
    reportingCharts.evolution = new Chart(ctx, {
        type: 'line',
        data: {
            labels: evolution.map(e => {
                const date = new Date(e.mois + '-01');
                return date.toLocaleDateString('fr-FR', { month: 'short', year: 'numeric' });
            }),
            datasets: [
                {
                    label: 'Ventes',
                    data: evolution.map(e => e.ventes),
                    borderColor: '#2196F3',
                    backgroundColor: 'rgba(33, 150, 243, 0.1)',
                    tension: 0.4,
                    fill: true
                },
                {
                    label: 'Achats',
                    data: evolution.map(e => e.achats),
                    borderColor: '#FF9800',
                    backgroundColor: 'rgba(255, 152, 0, 0.1)',
                    tension: 0.4,
                    fill: true
                },
                {
                    label: 'Marge',
                    data: evolution.map(e => e.marge),
                    borderColor: '#4CAF50',
                    backgroundColor: 'rgba(76, 175, 80, 0.1)',
                    tension: 0.4,
                    fill: true
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'top',
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                    callbacks: {
                        label: function(context) {
                            return `${context.dataset.label}: ${formatCurrency(context.parsed.y)}`;
                        }
                    }
                }
            },
            scales: {
                x: {
                    grid: {
                        display: false
                    }
                },
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return formatCurrency(value);
                        }
                    }
                }
            }
        }
    });
}

// Création du graphique des top sites
function createTopSitesChart(topSites) {
    const ctx = document.getElementById('sites-performance-chart');
    if (!ctx) return;
    
    // Détruire le graphique existant
    if (reportingCharts.sites) {
        reportingCharts.sites.destroy();
    }
    
    const top5Sites = topSites.slice(0, 5);
    
    reportingCharts.sites = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: top5Sites.map(s => s.nom),
            datasets: [
                {
                    label: 'Ventes',
                    data: top5Sites.map(s => s.ventes_mois),
                    backgroundColor: 'rgba(33, 150, 243, 0.8)',
                    borderColor: '#2196F3',
                    borderWidth: 1
                },
                {
                    label: 'Marge',
                    data: top5Sites.map(s => s.marge_mois),
                    backgroundColor: 'rgba(76, 175, 80, 0.8)',
                    borderColor: '#4CAF50',
                    borderWidth: 1
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'top',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `${context.dataset.label}: ${formatCurrency(context.parsed.y)}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return formatCurrency(value);
                        }
                    }
                }
            }
        }
    });
}

// Affichage du tableau des top sites
function displayTopSitesTable(topSites) {
    const tableContainer = document.querySelector('#top-sites-table');
    if (!tableContainer) return;
    
    const table = `
        <table>
            <thead>
                <tr>
                    <th>Rang</th>
                    <th>Site</th>
                    <th>Ventes</th>
                    <th>Marge</th>
                    <th>ROI</th>
                </tr>
            </thead>
            <tbody>
                ${topSites.map((site, index) => {
                    const roi = site.ventes_mois > 0 ? (site.marge_mois / site.ventes_mois) * 100 : 0;
                    return `
                        <tr>
                            <td>${index + 1}</td>
                            <td>${site.nom}</td>
                            <td>${formatCurrency(site.ventes_mois)}</td>
                            <td>${formatCurrency(site.marge_mois)}</td>
                            <td>${roi.toFixed(1)}%</td>
                        </tr>
                    `;
                }).join('')}
            </tbody>
        </table>
    `;
    
    tableContainer.innerHTML = table;
}

// Initialisation des contrôles de reporting
function initReportingControls() {
    // Filtres de période
    const periodeSelect = document.getElementById('reporting-periode');
    const dateDebutInput = document.getElementById('reporting-date-debut');
    const dateFinInput = document.getElementById('reporting-date-fin');
    
    if (periodeSelect) {
        periodeSelect.addEventListener('change', handlePeriodeChange);
    }
    
    if (dateDebutInput && dateFinInput) {
        [dateDebutInput, dateFinInput].forEach(input => {
            input.addEventListener('change', handleDateChange);
        });
    }
    
    // Boutons d'export
    const exportPdfBtn = document.getElementById('export-pdf');
    const exportExcelBtn = document.getElementById('export-excel');
    
    if (exportPdfBtn) {
        exportPdfBtn.addEventListener('click', exportToPDF);
    }
    
    if (exportExcelBtn) {
        exportExcelBtn.addEventListener('click', exportToExcel);
    }
}

// Gestion du changement de période
function handlePeriodeChange(event) {
    const periode = event.target.value;
    const today = new Date();
    let dateDebut, dateFin;
    
    switch (periode) {
        case 'semaine':
            dateDebut = new Date(today.getFullYear(), today.getMonth(), today.getDate() - 7);
            dateFin = today;
            break;
        case 'mois':
            dateDebut = new Date(today.getFullYear(), today.getMonth(), 1);
            dateFin = today;
            break;
        case 'trimestre':
            dateDebut = new Date(today.getFullYear(), Math.floor(today.getMonth() / 3) * 3, 1);
            dateFin = today;
            break;
        case 'annee':
            dateDebut = new Date(today.getFullYear(), 0, 1);
            dateFin = today;
            break;
        default:
            return;
    }
    
    // Mettre à jour les champs de date
    const dateDebutInput = document.getElementById('reporting-date-debut');
    const dateFinInput = document.getElementById('reporting-date-fin');
    
    if (dateDebutInput) dateDebutInput.value = dateDebut.toISOString().substring(0, 10);
    if (dateFinInput) dateFin.value = dateFin.toISOString().substring(0, 10);
    
    // Recharger les données
    loadReportForPeriod(dateDebut, dateFin);
}

// Gestion du changement de dates
function handleDateChange() {
    const dateDebut = document.getElementById('reporting-date-debut')?.value;
    const dateFin = document.getElementById('reporting-date-fin')?.value;
    
    if (dateDebut && dateFin) {
        loadReportForPeriod(new Date(dateDebut), new Date(dateFin));
    }
}

// Chargement des données pour une période spécifique
async function loadReportForPeriod(dateDebut, dateFin) {
    try {
        showReportingLoading(true);
        
        const dateDebutStr = dateDebut.toISOString().substring(0, 10);
        const dateFinStr = dateFin.toISOString().substring(0, 10);
        
        if (isOnline) {
            const response = await fetch(`/api/reporting/rentabilite?date_debut=${dateDebutStr}&date_fin=${dateFinStr}`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                currentReportData = data.data;
                displayPeriodReport(currentReportData);
            }
        } else {
            // Générer le rapport local pour la période
            currentReportData = await generateLocalReportForPeriod(dateDebutStr, dateFinStr);
            displayPeriodReport(currentReportData);
        }
        
        showReportingLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement rapport période:', error);
        showError('Erreur lors du chargement du rapport');
        showReportingLoading(false);
    }
}

// Export PDF (fonctionnalité de base)
function exportToPDF() {
    showNotification('Fonctionnalité d\'export PDF en cours de développement', 'info');
}

// Export Excel (fonctionnalité de base)
function exportToExcel() {
    if (!currentReportData) {
        showError('Aucune donnée à exporter');
        return;
    }
    
    // Création d'un CSV simple
    let csvContent = "data:text/csv;charset=utf-8,";
    csvContent += "Site,Ventes,Marge,ROI\n";
    
    if (currentReportData.par_site) {
        currentReportData.par_site.forEach(site => {
            csvContent += `"${site.site_nom}",${site.total_ventes},${site.marge_brute},${site.roi_pourcentage}\n`;
        });
    }
    
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "rapport_rentabilite.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    showNotification('Rapport exporté en CSV', 'success');
}

function showReportingLoading(show) {
    const reportingContainer = document.querySelector('#reporting-container');
    if (!reportingContainer) return;
    
    if (show) {
        const loadingOverlay = document.createElement('div');
        loadingOverlay.className = 'loading-overlay';
        loadingOverlay.innerHTML = `
            <div class="loading-spinner"></div>
            <p>Génération du rapport...</p>
        `;
        reportingContainer.appendChild(loadingOverlay);
    } else {
        const loadingOverlay = reportingContainer.querySelector('.loading-overlay');
        if (loadingOverlay) {
            loadingOverlay.remove();
        }
    }
}

// Export des fonctions
window.loadReportingData = loadReportingData;

//////////////////////////////////////////////
// js/entities.js - Gestion des entités de base (produits, fournisseurs, clients)
//////////////////////////////////////////////

// Fonctions pour la gestion des produits
async function loadProduitsData() {
    try {
        const response = await fetch('/api/produits', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            displayProduitsTable(data.data);
        } else {
            const localProduits = await dbManager.readAll('produits');
            displayProduitsTable(localProduits);
        }
    } catch (error) {
        console.error('Erreur chargement produits:', error);
        const localProduits = await dbManager.readAll('produits');
        displayProduitsTable(localProduits);
    }
}

// Fonctions pour la gestion des fournisseurs
async function loadFournisseursData() {
    try {
        const response = await fetch('/api/fournisseurs', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            displayFournisseursTable(data.data);
        } else {
            const localFournisseurs = await dbManager.readAll('fournisseurs');
            displayFournisseursTable(localFournisseurs);
        }
    } catch (error) {
        console.error('Erreur chargement fournisseurs:', error);
        const localFournisseurs = await dbManager.readAll('fournisseurs');
        displayFournisseursTable(localFournisseurs);
    }
}

// Fonctions pour la gestion des clients
async function loadClientsData() {
    try {
        const response = await fetch('/api/clients', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            displayClientsTable(data.data);
        } else {
            const localClients = await dbManager.readAll('clients');
            displayClientsTable(localClients);
        }
    } catch (error) {
        console.error('Erreur chargement clients:', error);
        const localClients = await dbManager.readAll('clients');
        displayClientsTable(localClients);
    }
}

// Fonctions utilitaires
function groupBy(array, key) {
    return array.reduce((result, item) => {
        const group = item[key];
        if (!result[group]) {
            result[group] = [];
        }
        result[group].push(item);
        return result;
    }, {});
}

// Export des fonctions
window.loadProduitsData = loadProduitsData;
window.loadFournisseursData = loadFournisseursData;
window.loadClientsData = loadClientsData;