// js/sites.js - Gestion des sites et budgets

// Variables globales pour les sites
let currentSites = [];
let selectedSite = null;

// Chargement des données des sites
async function loadSitesData() {
    try {
        showSitesLoading(true);
        
        // Charger les sites avec leurs budgets
        const sitesWithBudgets = await dbManager.getSitesBudgets();
        currentSites = sitesWithBudgets;
        
        // Afficher les sites
        displaySitesGrid(sitesWithBudgets);
        
        // Charger les statistiques globales
        await loadSitesStatistics();
        
        showSitesLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement sites:', error);
        showError('Erreur lors du chargement des sites');
        showSitesLoading(false);
    }
}

// Affichage de la grille des sites
function displaySitesGrid(sites) {
    const sitesGrid = document.getElementById('sites-grid');
    if (!sitesGrid) return;
    
    sitesGrid.innerHTML = '';
    
    if (sites.length === 0) {
        sitesGrid.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-map-marker-alt"></i>
                <h3>Aucun site enregistré</h3>
                <p>Commencez par créer votre premier site d'achat</p>
                <button class="btn btn-primary" onclick="openModal('modal-site')">
                    <i class="fas fa-plus"></i> Créer un site
                </button>
            </div>
        `;
        return;
    }
    
    sites.forEach(site => {
        const siteCard = createSiteCard(site);
        sitesGrid.appendChild(siteCard);
    });
}

// Création d'une carte de site
function createSiteCard(site) {
    const card = document.createElement('div');
    card.className = 'site-card card';
    card.dataset.siteId = site.id;
    
    const statusClass = getSiteStatusClass(site.actif);
    const typeIcon = getSiteTypeIcon(site.type_site);
    const budgetPercentage = calculateBudgetUsagePercentage(site);
    
    card.innerHTML = `
        <div class="site-card-header">
            <div class="site-status ${statusClass}"></div>
            <div class="site-info">
                <h3>${site.nom}</h3>
                <p><i class="fas ${typeIcon}"></i> ${formatSiteType(site.type_site)}</p>
            </div>
        </div>
        
        <div class="card-body">
            <div class="site-metrics">
                <div class="metric">
                    <div class="metric-value">${formatCurrency(site.budget_total)}</div>
                    <div class="metric-label">Budget disponible</div>
                </div>
                <div class="metric">
                    <div class="metric-value">${site.budgets.length}</div>
                    <div class="metric-label">Allocations</div>
                </div>
                <div class="metric">
                    <div class="metric-value" id="site-${site.id}-achats">-</div>
                    <div class="metric-label">Achats ce mois</div>
                </div>
                <div class="metric">
                    <div class="metric-value" id="site-${site.id}-ventes">-</div>
                    <div class="metric-label">Ventes ce mois</div>
                </div>
            </div>
            
            <div class="budget-progress">
                <div class="progress-header">
                    <span>Utilisation budget</span>
                    <span>${budgetPercentage.toFixed(1)}%</span>
                </div>
                <div class="progress">
                    <div class="progress-bar" style="width: ${Math.min(budgetPercentage, 100)}%"></div>
                </div>
            </div>
            
            ${site.adresse ? `
                <div class="site-address">
                    <i class="fas fa-map-marker-alt"></i>
                    <span>${site.adresse}</span>
                </div>
            ` : ''}
        </div>
        
        <div class="card-footer">
            <div class="site-actions">
                <button class="btn btn-sm btn-outline" onclick="viewSiteDetails(${site.id})">
                    <i class="fas fa-eye"></i> Détails
                </button>
                <button class="btn btn-sm btn-primary" onclick="manageBudget(${site.id})">
                    <i class="fas fa-wallet"></i> Budget
                </button>
                <div class="dropdown">
                    <button class="btn btn-sm btn-ghost dropdown-toggle" onclick="toggleSiteMenu(${site.id})">
                        <i class="fas fa-ellipsis-v"></i>
                    </button>
                    <div class="dropdown-menu" id="site-menu-${site.id}">
                        <a href="#" class="dropdown-item" onclick="editSite(${site.id})">
                            <i class="fas fa-edit"></i> Modifier
                        </a>
                        <a href="#" class="dropdown-item" onclick="viewSiteReports(${site.id})">
                            <i class="fas fa-chart-bar"></i> Rapports
                        </a>
                        <div class="dropdown-divider"></div>
                        <a href="#" class="dropdown-item text-danger" onclick="deleteSite(${site.id})">
                            <i class="fas fa-trash"></i> Supprimer
                        </a>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Charger les métriques asynchrones
    loadSiteMetrics(site.id);
    
    // Ajouter les événements
    card.addEventListener('click', (e) => {
        if (!e.target.closest('.site-actions')) {
            viewSiteDetails(site.id);
        }
    });
    
    return card;
}

// Chargement des métriques d'un site
async function loadSiteMetrics(siteId) {
    try {
        const currentMonth = new Date().toISOString().substring(0, 7);
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        // Achats du mois
        const achats = await dbManager.readAll('achats', 'site_id', siteId);
        const achatsMonth = achats.filter(a => 
            a.date_achat >= startOfMonth && a.date_achat <= endOfMonth
        );
        const totalAchats = achatsMonth.reduce((sum, a) => sum + a.cout_total, 0);
        
        // Ventes du mois
        const ventes = await dbManager.readAll('ventes', 'site_id', siteId);
        const ventesMonth = ventes.filter(v => 
            v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
        );
        const totalVentes = ventesMonth.reduce((sum, v) => sum + v.montant_total, 0);
        
        // Mettre à jour l'affichage
        const achatsElement = document.getElementById(`site-${siteId}-achats`);
        const ventesElement = document.getElementById(`site-${siteId}-ventes`);
        
        if (achatsElement) {
            achatsElement.textContent = formatCurrency(totalAchats);
        }
        
        if (ventesElement) {
            ventesElement.textContent = formatCurrency(totalVentes);
        }
        
    } catch (error) {
        console.error(`Erreur chargement métriques site ${siteId}:`, error);
    }
}

// Fonctions utilitaires pour les sites
function getSiteStatusClass(actif) {
    return actif ? 'active' : 'inactive';
}

function getSiteTypeIcon(type) {
    const icons = {
        'pont_bascule': 'fa-weight-hanging',
        'magasin_stockage': 'fa-warehouse',
        'autre': 'fa-building'
    };
    return icons[type] || 'fa-building';
}

function formatSiteType(type) {
    const types = {
        'pont_bascule': 'Pont bascule',
        'magasin_stockage': 'Magasin de stockage',
        'autre': 'Autre'
    };
    return types[type] || type;
}

function calculateBudgetUsagePercentage(site) {
    if (!site.budgets || site.budgets.length === 0) return 0;
    
    const totalInitial = site.budgets.reduce((sum, b) => sum + b.montant_initial, 0);
    const totalActuel = site.budgets.reduce((sum, b) => sum + b.montant_actuel, 0);
    
    if (totalInitial === 0) return 0;
    
    return ((totalInitial - totalActuel) / totalInitial) * 100;
}

// Gestion du formulaire de site
async function handleSiteSubmit(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    
    // Validation
    const errors = validateForm(form);
    if (errors.length > 0) {
        displayFormErrors(form, errors);
        return;
    }
    
    try {
        const siteData = {
            nom: formData.get('nom'),
            type_site: formData.get('type_site'),
            adresse: formData.get('adresse'),
            responsable_id: currentUser ? currentUser.id : null,
            actif: true
        };
        
        // Créer le site
        const siteId = await dbManager.create('sites', siteData);
        
        // Créer le budget initial si spécifié
        const budgetInitial = parseFloat(formData.get('budget_initial'));
        if (budgetInitial && budgetInitial > 0) {
            const budgetData = {
                site_id: siteId,
                montant_initial: budgetInitial,
                montant_actuel: budgetInitial,
                periode_debut: new Date().toISOString().substring(0, 10),
                periode_fin: new Date(Date.now() + 365 * 24 * 60 * 60 * 1000).toISOString().substring(0, 10),
                statut: 'actif'
            };
            
            await dbManager.create('budgets', budgetData);
        }
        
        // Fermer le modal et recharger
        closeModal('modal-site');
        showNotification('Site créé avec succès', 'success');
        await loadSitesData();
        
    } catch (error) {
        console.error('Erreur création site:', error);
        showError('Erreur lors de la création du site');
    }
}

// Affichage des détails d'un site
async function viewSiteDetails(siteId) {
    try {
        const site = currentSites.find(s => s.id === siteId);
        if (!site) {
            showError('Site non trouvé');
            return;
        }
        
        selectedSite = site;
        
        // Créer et afficher le modal de détails
        await showSiteDetailsModal(site);
        
    } catch (error) {
        console.error('Erreur affichage détails site:', error);
        showError('Erreur lors de l\'affichage des détails');
    }
}

async function showSiteDetailsModal(site) {
    const modal = document.createElement('div');
    modal.className = 'modal show';
    modal.id = 'modal-site-details';
    
    // Calculer les statistiques détaillées
    const stats = await calculateSiteDetailedStats(site.id);
    
    modal.innerHTML = `
        <div class="modal-content modal-lg">
            <div class="modal-header">
                <h3>${site.nom} - Détails</h3>
                <button class="modal-close" onclick="closeSiteDetailsModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="modal-body">
                <div class="site-details-tabs">
                    <ul class="tab-list">
                        <li class="tab-item">
                            <a href="#" class="tab-link active" onclick="showSiteTab('overview')">Vue d'ensemble</a>
                        </li>
                        <li class="tab-item">
                            <a href="#" class="tab-link" onclick="showSiteTab('budget')">Budget</a>
                        </li>
                        <li class="tab-item">
                            <a href="#" class="tab-link" onclick="showSiteTab('activity')">Activités</a>
                        </li>
                        <li class="tab-item">
                            <a href="#" class="tab-link" onclick="showSiteTab('stocks')">Stocks</a>
                        </li>
                    </ul>
                </div>
                
                <div class="tab-content-area">
                    <!-- Vue d'ensemble -->
                    <div id="site-tab-overview" class="tab-content active">
                        <div class="site-overview">
                            <div class="overview-grid">
                                <div class="overview-card">
                                    <h4>Informations générales</h4>
                                    <div class="info-list">
                                        <div class="info-item">
                                            <span class="label">Type:</span>
                                            <span class="value">${formatSiteType(site.type_site)}</span>
                                        </div>
                                        <div class="info-item">
                                            <span class="label">Statut:</span>
                                            <span class="value">
                                                <span class="badge ${site.actif ? 'badge-success' : 'badge-danger'}">
                                                    ${site.actif ? 'Actif' : 'Inactif'}
                                                </span>
                                            </span>
                                        </div>
                                        <div class="info-item">
                                            <span class="label">Adresse:</span>
                                            <span class="value">${site.adresse || 'Non renseignée'}</span>
                                        </div>
                                        <div class="info-item">
                                            <span class="label">Créé le:</span>
                                            <span class="value">${formatDate(site.date_creation, 'long')}</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="overview-card">
                                    <h4>Performance ce mois</h4>
                                    <div class="metrics-grid">
                                        <div class="metric-box">
                                            <div class="metric-value">${formatCurrency(stats.achats_mois)}</div>
                                            <div class="metric-label">Achats</div>
                                        </div>
                                        <div class="metric-box">
                                            <div class="metric-value">${formatCurrency(stats.ventes_mois)}</div>
                                            <div class="metric-label">Ventes</div>
                                        </div>
                                        <div class="metric-box">
                                            <div class="metric-value">${formatCurrency(stats.marge_mois)}</div>
                                            <div class="metric-label">Marge</div>
                                        </div>
                                        <div class="metric-box">
                                            <div class="metric-value">${stats.roi_mois.toFixed(1)}%</div>
                                            <div class="metric-label">ROI</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Budget -->
                    <div id="site-tab-budget" class="tab-content">
                        <div class="budget-management">
                            <div class="budget-header">
                                <h4>Gestion du budget</h4>
                                <button class="btn btn-primary" onclick="addBudgetAllocation(${site.id})">
                                    <i class="fas fa-plus"></i> Nouvelle allocation
                                </button>
                            </div>
                            
                            <div class="budget-list" id="site-budget-list">
                                <!-- Chargé dynamiquement -->
                            </div>
                        </div>
                    </div>
                    
                    <!-- Activités -->
                    <div id="site-tab-activity" class="tab-content">
                        <div class="activity-timeline" id="site-activity-timeline">
                            <!-- Chargé dynamiquement -->
                        </div>
                    </div>
                    
                    <!-- Stocks -->
                    <div id="site-tab-stocks" class="tab-content">
                        <div class="stocks-overview" id="site-stocks-overview">
                            <!-- Chargé dynamiquement -->
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="modal-footer">
                <button class="btn btn-secondary" onclick="closeSiteDetailsModal()">Fermer</button>
                <button class="btn btn-primary" onclick="editSite(${site.id})">
                    <i class="fas fa-edit"></i> Modifier
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.body.style.overflow = 'hidden';
    
    // Charger les données des onglets
    await loadSiteBudgetData(site.id);
    await loadSiteActivityData(site.id);
    await loadSiteStocksData(site.id);
}

// Calcul des statistiques détaillées d'un site
async function calculateSiteDetailedStats(siteId) {
    const currentMonth = new Date().toISOString().substring(0, 7);
    const startOfMonth = `${currentMonth}-01`;
    const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
        .toISOString().substring(0, 10);
    
    const achats = await dbManager.readAll('achats', 'site_id', siteId);
    const ventes = await dbManager.readAll('ventes', 'site_id', siteId);
    
    const achatsMonth = achats.filter(a => 
        a.date_achat >= startOfMonth && a.date_achat <= endOfMonth
    );
    const ventesMonth = ventes.filter(v => 
        v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
    );
    
    const achats_mois = achatsMonth.reduce((sum, a) => sum + a.cout_total, 0);
    const ventes_mois = ventesMonth.reduce((sum, v) => sum + v.montant_total, 0);
    const marge_mois = ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0);
    const roi_mois = achats_mois > 0 ? (marge_mois / achats_mois) * 100 : 0;
    
    return {
        achats_mois,
        ventes_mois,
        marge_mois,
        roi_mois
    };
}

// Gestion des onglets dans le modal de détails
function showSiteTab(tabName) {
    // Masquer tous les onglets
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    document.querySelectorAll('.tab-link').forEach(link => {
        link.classList.remove('active');
    });
    
    // Afficher l'onglet sélectionné
    const targetTab = document.getElementById(`site-tab-${tabName}`);
    const targetLink = document.querySelector(`[onclick="showSiteTab('${tabName}')"]`);
    
    if (targetTab) targetTab.classList.add('active');
    if (targetLink) targetLink.classList.add('active');
}

// Fermeture du modal de détails
function closeSiteDetailsModal() {
    const modal = document.getElementById('modal-site-details');
    if (modal) {
        modal.remove();
        document.body.style.overflow = '';
    }
}

// Chargement des statistiques globales des sites
async function loadSitesStatistics() {
    try {
        // Calculer les statistiques globales
        const totalSites = currentSites.length;
        const sitesActifs = currentSites.filter(s => s.actif).length;
        const budgetTotal = currentSites.reduce((sum, s) => sum + s.budget_total, 0);
        
        // Afficher les statistiques si il y a un conteneur dédié
        const statsContainer = document.querySelector('.sites-statistics');
        if (statsContainer) {
            statsContainer.innerHTML = `
                <div class="stat-item">
                    <div class="stat-value">${totalSites}</div>
                    <div class="stat-label">Sites total</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">${sitesActifs}</div>
                    <div class="stat-label">Sites actifs</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">${formatCurrency(budgetTotal)}</div>
                    <div class="stat-label">Budget total</div>
                </div>
            `;
        }
        
    } catch (error) {
        console.error('Erreur calcul statistiques sites:', error);
    }
}

// Affichage du loading
function showSitesLoading(show) {
    const sitesGrid = document.getElementById('sites-grid');
    if (!sitesGrid) return;
    
    if (show) {
        sitesGrid.innerHTML = `
            <div class="loading-container">
                <div class="loading-spinner"></div>
                <p>Chargement des sites...</p>
            </div>
        `;
    }
}

// Fonctions à implémenter
function toggleSiteMenu(siteId) {
    const menu = document.getElementById(`site-menu-${siteId}`);
    if (menu) {
        menu.classList.toggle('show');
    }
}

function editSite(siteId) {
    console.log('Édition site:', siteId);
    // À implémenter
}

function manageBudget(siteId) {
    console.log('Gestion budget site:', siteId);
    // À implémenter
}

function viewSiteReports(siteId) {
    console.log('Rapports site:', siteId);
    // À implémenter
}

function deleteSite(siteId) {
    if (confirm('Êtes-vous sûr de vouloir supprimer ce site ?')) {
        console.log('Suppression site:', siteId);
        // À implémenter
    }
}

// Chargement des données pour les onglets du modal
async function loadSiteBudgetData(siteId) {
    console.log('Chargement budget site:', siteId);
    // À implémenter
}

async function loadSiteActivityData(siteId) {
    console.log('Chargement activité site:', siteId);
    // À implémenter
}

async function loadSiteStocksData(siteId) {
    console.log('Chargement stocks site:', siteId);
    // À implémenter
}

// Export des fonctions
window.loadSitesData = loadSitesData;
window.handleSiteSubmit = handleSiteSubmit;
window.viewSiteDetails = viewSiteDetails;
window.showSiteTab = showSiteTab;
window.closeSiteDetailsModal = closeSiteDetailsModal;
window.toggleSiteMenu = toggleSiteMenu;
window.editSite = editSite;
window.manageBudget = manageBudget;
window.viewSiteReports = viewSiteReports;
window.deleteSite = deleteSite;