// database.js - Gestion de la base de données locale IndexedDB
class DatabaseManager {
    constructor() {
        this.dbName = 'GestionFinancementDB';
        this.dbVersion = 1;
        this.db = null;
    }

    // Initialisation de la base de données IndexedDB
    async init() {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(this.dbName, this.dbVersion);

            request.onerror = () => reject(request.error);
            request.onsuccess = () => {
                this.db = request.result;
                resolve(this.db);
            };

            request.onupgradeneeded = (event) => {
                const db = event.target.result;
                this.createStores(db);
            };
        });
    }

    // Création des stores (tables) IndexedDB
    createStores(db) {
        // Store pour les utilisateurs
        if (!db.objectStoreNames.contains('utilisateurs')) {
            const userStore = db.createObjectStore('utilisateurs', { keyPath: 'id', autoIncrement: true });
            userStore.createIndex('email', 'email', { unique: true });
        }

        // Store pour les sites
        if (!db.objectStoreNames.contains('sites')) {
            const siteStore = db.createObjectStore('sites', { keyPath: 'id', autoIncrement: true });
            siteStore.createIndex('nom', 'nom');
        }

        // Store pour les budgets
        if (!db.objectStoreNames.contains('budgets')) {
            const budgetStore = db.createObjectStore('budgets', { keyPath: 'id', autoIncrement: true });
            budgetStore.createIndex('site_id', 'site_id');
        }

        // Store pour les fournisseurs
        if (!db.objectStoreNames.contains('fournisseurs')) {
            const fournisseurStore = db.createObjectStore('fournisseurs', { keyPath: 'id', autoIncrement: true });
            fournisseurStore.createIndex('nom', 'nom');
        }

        // Store pour les produits
        if (!db.objectStoreNames.contains('produits')) {
            const produitStore = db.createObjectStore('produits', { keyPath: 'id', autoIncrement: true });
            produitStore.createIndex('nom', 'nom');
        }

        // Store pour les achats
        if (!db.objectStoreNames.contains('achats')) {
            const achatStore = db.createObjectStore('achats', { keyPath: 'id', autoIncrement: true });
            achatStore.createIndex('site_id', 'site_id');
            achatStore.createIndex('date_achat', 'date_achat');
        }

        // Store pour les stocks
        if (!db.objectStoreNames.contains('stocks')) {
            const stockStore = db.createObjectStore('stocks', { keyPath: 'id', autoIncrement: true });
            stockStore.createIndex('site_produit', ['site_id', 'produit_id'], { unique: true });
        }

        // Store pour les ventes
        if (!db.objectStoreNames.contains('ventes')) {
            const venteStore = db.createObjectStore('ventes', { keyPath: 'id', autoIncrement: true });
            venteStore.createIndex('site_id', 'site_id');
            venteStore.createIndex('date_vente', 'date_vente');
        }

        // Store pour les clients
        if (!db.objectStoreNames.contains('clients')) {
            const clientStore = db.createObjectStore('clients', { keyPath: 'id', autoIncrement: true });
            clientStore.createIndex('nom', 'nom');
        }

        // Store pour les transactions
        if (!db.objectStoreNames.contains('transactions')) {
            const transactionStore = db.createObjectStore('transactions', { keyPath: 'id', autoIncrement: true });
            transactionStore.createIndex('site_id', 'site_id');
            transactionStore.createIndex('date_transaction', 'date_transaction');
        }

        // Store pour le statut de synchronisation
        if (!db.objectStoreNames.contains('sync_queue')) {
            const syncStore = db.createObjectStore('sync_queue', { keyPath: 'id', autoIncrement: true });
            syncStore.createIndex('synced', 'synced');
        }
    }

    // Méthodes CRUD génériques
    async create(storeName, data) {
        const transaction = this.db.transaction([storeName], 'readwrite');
        const store = transaction.objectStore(storeName);
        
        // Ajouter les métadonnées
        data.date_creation = new Date().toISOString();
        data.last_modified = new Date().toISOString();
        data.synced = false;

        const result = await this.promisifyRequest(store.add(data));
        
        // Ajouter à la queue de synchronisation
        await this.addToSyncQueue(storeName, result, 'insert', data);
        
        return result;
    }

    async read(storeName, id) {
        const transaction = this.db.transaction([storeName], 'readonly');
        const store = transaction.objectStore(storeName);
        return await this.promisifyRequest(store.get(id));
    }

    async readAll(storeName, indexName = null, query = null) {
        const transaction = this.db.transaction([storeName], 'readonly');
        const store = transaction.objectStore(storeName);
        
        if (indexName && query) {
            const index = store.index(indexName);
            return await this.promisifyRequest(index.getAll(query));
        }
        
        return await this.promisifyRequest(store.getAll());
    }

    async update(storeName, data) {
        data.last_modified = new Date().toISOString();
        data.synced = false;

        const transaction = this.db.transaction([storeName], 'readwrite');
        const store = transaction.objectStore(storeName);
        
        const result = await this.promisifyRequest(store.put(data));
        
        // Ajouter à la queue de synchronisation
        await this.addToSyncQueue(storeName, data.id, 'update', data);
        
        return result;
    }

    async delete(storeName, id) {
        const transaction = this.db.transaction([storeName], 'readwrite');
        const store = transaction.objectStore(storeName);
        
        const result = await this.promisifyRequest(store.delete(id));
        
        // Ajouter à la queue de synchronisation
        await this.addToSyncQueue(storeName, id, 'delete', { id });
        
        return result;
    }

    // Ajouter un élément à la queue de synchronisation
    async addToSyncQueue(tableName, recordId, action, data) {
        const syncItem = {
            table_name: tableName,
            record_id: recordId,
            action: action,
            data: JSON.stringify(data),
            synced: false,
            date_creation: new Date().toISOString()
        };

        const transaction = this.db.transaction(['sync_queue'], 'readwrite');
        const store = transaction.objectStore('sync_queue');
        return await this.promisifyRequest(store.add(syncItem));
    }

    // Utilitaire pour promisifier les requêtes IndexedDB
    promisifyRequest(request) {
        return new Promise((resolve, reject) => {
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    }

    // Méthodes spécifiques métier
    async getSitesBudgets() {
        const sites = await this.readAll('sites');
        const budgets = await this.readAll('budgets');
        
        return sites.map(site => {
            const siteBudgets = budgets.filter(b => b.site_id === site.id);
            return {
                ...site,
                budgets: siteBudgets,
                budget_total: siteBudgets.reduce((sum, b) => sum + b.montant_actuel, 0)
            };
        });
    }

    async getStocksSite(siteId) {
        const stocks = await this.readAll('stocks', 'site_id', siteId);
        const produits = await this.readAll('produits');
        
        return stocks.map(stock => {
            const produit = produits.find(p => p.id === stock.produit_id);
            return {
                ...stock,
                produit_nom: produit ? produit.nom : 'Produit inconnu',
                unite_mesure: produit ? produit.unite_mesure : ''
            };
        });
    }

    async calculerRentabiliteSite(siteId, dateDebut, dateFin) {
        const ventes = await this.readAll('ventes', 'site_id', siteId);
        const achats = await this.readAll('achats', 'site_id', siteId);
        
        // Filtrer par période
        const ventesFiltered = ventes.filter(v => 
            v.date_vente >= dateDebut && v.date_vente <= dateFin
        );
        const achatsFiltered = achats.filter(a => 
            a.date_achat >= dateDebut && a.date_achat <= dateFin
        );

        const totalVentes = ventesFiltered.reduce((sum, v) => sum + v.montant_total, 0);
        const totalAchats = achatsFiltered.reduce((sum, a) => sum + a.cout_total, 0);
        const margeBrute = ventesFiltered.reduce((sum, v) => sum + v.marge_brute, 0);

        return {
            total_ventes: totalVentes,
            total_achats: totalAchats,
            marge_brute: margeBrute,
            roi: totalAchats > 0 ? (margeBrute / totalAchats) * 100 : 0
        };
    }
}

// Gestionnaire de synchronisation
class SyncManager {
    constructor(databaseManager, apiUrl) {
        this.db = databaseManager;
        this.apiUrl = apiUrl;
        this.isOnline = navigator.onLine;
        
        // Écouter les changements de connectivité
        window.addEventListener('online', () => {
            this.isOnline = true;
            this.syncAll();
        });
        
        window.addEventListener('offline', () => {
            this.isOnline = false;
        });
    }

    async syncAll() {
        if (!this.isOnline) return;

        try {
            const syncItems = await this.db.readAll('sync_queue', 'synced', false);
            
            for (const item of syncItems) {
                await this.syncItem(item);
            }
        } catch (error) {
            console.error('Erreur lors de la synchronisation:', error);
        }
    }

    async syncItem(syncItem) {
        try {
            const url = `${this.apiUrl}/sync.php`;
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(syncItem)
            });

            if (response.ok) {
                // Marquer comme synchronisé
                syncItem.synced = true;
                syncItem.date_sync = new Date().toISOString();
                await this.db.update('sync_queue', syncItem);
            }
        } catch (error) {
            console.error('Erreur sync item:', error);
        }
    }

    async downloadData() {
        if (!this.isOnline) return;

        try {
            const response = await fetch(`${this.apiUrl}/download.php`);
            const data = await response.json();

            // Mettre à jour la base locale avec les données serveur
            for (const [tableName, records] of Object.entries(data)) {
                for (const record of records) {
                    record.synced = true;
                    await this.db.update(tableName, record);
                }
            }
        } catch (error) {
            console.error('Erreur téléchargement:', error);
        }
    }
}

// Instance globale
let dbManager = null;
let syncManager = null;

// Initialisation
async function initDatabase() {
    try {
        dbManager = new DatabaseManager();
        await dbManager.init();
        
        syncManager = new SyncManager(dbManager, '/api');
        
        console.log('Base de données initialisée');
        return true;
    } catch (error) {
        console.error('Erreur initialisation DB:', error);
        return false;
    }
}