// js/dashboard.js - Gestion du tableau de bord et KPI

// Variables pour les graphiques
let salesChart = null;
let sitesChart = null;

// Chargement des données du dashboard
async function loadDashboardData() {
    try {
        showDashboardLoading(true);
        
        // Charger les KPI
        await loadKPIData();
        
        // Charger les graphiques
        await loadChartsData();
        
        // Charger les transactions récentes
        await loadRecentTransactions();
        
        showDashboardLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement dashboard:', error);
        showError('Erreur lors du chargement du tableau de bord');
        showDashboardLoading(false);
    }
}

// Chargement des indicateurs clés (KPI)
async function loadKPIData() {
    try {
        const currentMonth = new Date().toISOString().substring(0, 7); // YYYY-MM
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        // Calculer le budget total
        const budgets = await dbManager.readAll('budgets');
        const totalBudget = budgets
            .filter(b => b.statut === 'actif')
            .reduce((sum, b) => sum + b.montant_actuel, 0);
        
        // Calculer les achats du mois
        const achats = await dbManager.readAll('achats');
        const achatsMonth = achats.filter(a => 
            a.date_achat >= startOfMonth && a.date_achat <= endOfMonth
        );
        const totalAchats = achatsMonth.reduce((sum, a) => sum + a.cout_total, 0);
        
        // Calculer les ventes du mois
        const ventes = await dbManager.readAll('ventes');
        const ventesMonth = ventes.filter(v => 
            v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
        );
        const totalVentes = ventesMonth.reduce((sum, v) => sum + v.montant_total, 0);
        
        // Calculer le ROI global
        const totalCouts = achatsMonth.reduce((sum, a) => sum + a.cout_total, 0);
        const totalMarges = ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0);
        const roiGlobal = totalCouts > 0 ? (totalMarges / totalCouts) * 100 : 0;
        
        // Mettre à jour l'interface
        updateKPIDisplay({
            totalBudget,
            totalAchats,
            totalVentes,
            roiGlobal
        });
        
        // Analyser les tendances
        await analyzeTrends();
        
    } catch (error) {
        console.error('Erreur chargement KPI:', error);
        throw error;
    }
}

// Mise à jour de l'affichage des KPI
function updateKPIDisplay(kpis) {
    // Budget total
    const budgetElement = document.getElementById('total-budget');
    if (budgetElement) {
        budgetElement.textContent = formatCurrency(kpis.totalBudget);
        animateNumber(budgetElement, 0, kpis.totalBudget, 1000);
    }
    
    // Achats du mois
    const achatsElement = document.getElementById('total-achats');
    if (achatsElement) {
        achatsElement.textContent = formatCurrency(kpis.totalAchats);
        animateNumber(achatsElement, 0, kpis.totalAchats, 1000);
    }
    
    // Ventes du mois
    const ventesElement = document.getElementById('total-ventes');
    if (ventesElement) {
        ventesElement.textContent = formatCurrency(kpis.totalVentes);
        animateNumber(ventesElement, 0, kpis.totalVentes, 1000);
    }
    
    // ROI Global
    const roiElement = document.getElementById('roi-global');
    if (roiElement) {
        roiElement.textContent = `${kpis.roiGlobal.toFixed(1)}%`;
        animateNumber(roiElement, 0, kpis.roiGlobal, 1000, '%');
        
        // Changer la couleur selon la performance
        const card = roiElement.closest('.kpi-card');
        if (card) {
            card.classList.remove('positive', 'negative', 'neutral');
            if (kpis.roiGlobal > 20) {
                card.classList.add('positive');
            } else if (kpis.roiGlobal < 10) {
                card.classList.add('negative');
            } else {
                card.classList.add('neutral');
            }
        }
    }
}

// Animation des nombres
function animateNumber(element, start, end, duration, suffix = '') {
    const startTime = performance.now();
    const difference = end - start;
    
    function updateNumber(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        
        // Fonction d'easing
        const easeOutQuart = 1 - Math.pow(1 - progress, 4);
        const currentValue = start + (difference * easeOutQuart);
        
        if (suffix === '%') {
            element.textContent = `${currentValue.toFixed(1)}%`;
        } else {
            element.textContent = formatCurrency(currentValue);
        }
        
        if (progress < 1) {
            requestAnimationFrame(updateNumber);
        }
    }
    
    requestAnimationFrame(updateNumber);
}

// Analyse des tendances
async function analyzeTrends() {
    try {
        const last6Months = Array.from({ length: 6 }, (_, i) => {
            const date = new Date();
            date.setMonth(date.getMonth() - i);
            return date.toISOString().substring(0, 7);
        }).reverse();
        
        const trends = {};
        
        for (const month of last6Months) {
            const startDate = `${month}-01`;
            const endDate = new Date(
                parseInt(month.split('-')[0]),
                parseInt(month.split('-')[1]),
                0
            ).toISOString().substring(0, 10);
            
            const achats = await dbManager.readAll('achats');
            const ventes = await dbManager.readAll('ventes');
            
            const achatsMonth = achats.filter(a => 
                a.date_achat >= startDate && a.date_achat <= endDate
            );
            const ventesMonth = ventes.filter(v => 
                v.date_vente >= startDate && v.date_vente <= endDate
            );
            
            trends[month] = {
                achats: achatsMonth.reduce((sum, a) => sum + a.cout_total, 0),
                ventes: ventesMonth.reduce((sum, v) => sum + v.montant_total, 0),
                marges: ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0)
            };
        }
        
        // Afficher les tendances dans l'interface
        displayTrends(trends);
        
    } catch (error) {
        console.error('Erreur analyse tendances:', error);
    }
}

// Affichage des tendances
function displayTrends(trends) {
    const months = Object.keys(trends);
    const currentMonth = months[months.length - 1];
    const previousMonth = months[months.length - 2];
    
    if (currentMonth && previousMonth) {
        const currentData = trends[currentMonth];
        const previousData = trends[previousMonth];
        
        // Calculer les variations
        const ventesVariation = calculateVariation(
            previousData.ventes, 
            currentData.ventes
        );
        const achatsVariation = calculateVariation(
            previousData.achats, 
            currentData.achats
        );
        
        // Afficher les indicateurs de tendance
        displayTrendIndicators({
            ventes: ventesVariation,
            achats: achatsVariation
        });
    }
}

function calculateVariation(previous, current) {
    if (previous === 0) return current > 0 ? 100 : 0;
    return ((current - previous) / previous) * 100;
}

function displayTrendIndicators(variations) {
    // Ajouter des indicateurs de tendance aux KPI
    const ventesCard = document.getElementById('total-ventes')?.closest('.kpi-card');
    const achatsCard = document.getElementById('total-achats')?.closest('.kpi-card');
    
    if (ventesCard) {
        addTrendIndicator(ventesCard, variations.ventes, 'ventes');
    }
    
    if (achatsCard) {
        addTrendIndicator(achatsCard, variations.achats, 'achats');
    }
}

function addTrendIndicator(card, variation, type) {
    // Supprimer l'indicateur existant
    const existingIndicator = card.querySelector('.trend-indicator');
    if (existingIndicator) {
        existingIndicator.remove();
    }
    
    const indicator = document.createElement('div');
    indicator.className = 'trend-indicator';
    
    const isPositive = variation > 0;
    const iconClass = isPositive ? 'fa-arrow-up' : 'fa-arrow-down';
    const colorClass = type === 'ventes' 
        ? (isPositive ? 'positive' : 'negative')
        : (isPositive ? 'negative' : 'positive'); // Pour les achats, moins c'est mieux
    
    indicator.innerHTML = `
        <i class="fas ${iconClass}"></i>
        <span>${Math.abs(variation).toFixed(1)}%</span>
    `;
    indicator.classList.add(colorClass);
    
    card.appendChild(indicator);
}

// Chargement des données pour les graphiques
async function loadChartsData() {
    try {
        await loadSalesChart();
        await loadSitesChart();
    } catch (error) {
        console.error('Erreur chargement graphiques:', error);
    }
}

// Graphique d'évolution des ventes
async function loadSalesChart() {
    const ctx = document.getElementById('sales-chart');
    if (!ctx) return;
    
    try {
        // Préparer les données des 12 derniers mois
        const last12Months = Array.from({ length: 12 }, (_, i) => {
            const date = new Date();
            date.setMonth(date.getMonth() - 11 + i);
            return {
                key: date.toISOString().substring(0, 7),
                label: date.toLocaleDateString('fr-FR', { month: 'short', year: 'numeric' })
            };
        });
        
        const ventes = await dbManager.readAll('ventes');
        const achats = await dbManager.readAll('achats');
        
        const salesData = last12Months.map(month => {
            const startDate = `${month.key}-01`;
            const endDate = new Date(
                parseInt(month.key.split('-')[0]),
                parseInt(month.key.split('-')[1]),
                0
            ).toISOString().substring(0, 10);
            
            const ventesMonth = ventes.filter(v => 
                v.date_vente >= startDate && v.date_vente <= endDate
            );
            const achatsMonth = achats.filter(a => 
                a.date_achat >= startDate && a.date_achat <= endDate
            );
            
            return {
                month: month.label,
                ventes: ventesMonth.reduce((sum, v) => sum + v.montant_total, 0),
                achats: achatsMonth.reduce((sum, a) => sum + a.cout_total, 0),
                marge: ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0)
            };
        });
        
        // Détruire le graphique existant
        if (salesChart) {
            salesChart.destroy();
        }
        
        // Créer le nouveau graphique
        salesChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: salesData.map(d => d.month),
                datasets: [
                    {
                        label: 'Ventes',
                        data: salesData.map(d => d.ventes),
                        borderColor: '#2196F3',
                        backgroundColor: 'rgba(33, 150, 243, 0.1)',
                        tension: 0.4,
                        fill: true
                    },
                    {
                        label: 'Achats',
                        data: salesData.map(d => d.achats),
                        borderColor: '#FF9800',
                        backgroundColor: 'rgba(255, 152, 0, 0.1)',
                        tension: 0.4,
                        fill: true
                    },
                    {
                        label: 'Marge',
                        data: salesData.map(d => d.marge),
                        borderColor: '#4CAF50',
                        backgroundColor: 'rgba(76, 175, 80, 0.1)',
                        tension: 0.4,
                        fill: true
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: function(context) {
                                return `${context.dataset.label}: ${formatCurrency(context.parsed.y)}`;
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        grid: {
                            display: false
                        }
                    },
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return formatCurrency(value);
                            }
                        }
                    }
                },
                interaction: {
                    mode: 'nearest',
                    axis: 'x',
                    intersect: false
                }
            }
        });
        
    } catch (error) {
        console.error('Erreur création graphique ventes:', error);
    }
}

// Graphique de répartition par site
async function loadSitesChart() {
    const ctx = document.getElementById('sites-chart');
    if (!ctx) return;
    
    try {
        const sites = await dbManager.readAll('sites');
        const ventes = await dbManager.readAll('ventes');
        
        // Calculer les ventes par site (dernier mois)
        const currentMonth = new Date().toISOString().substring(0, 7);
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        const ventesMonth = ventes.filter(v => 
            v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
        );
        
        const siteData = sites.map(site => {
            const siteVentes = ventesMonth.filter(v => v.site_id === site.id);
            const total = siteVentes.reduce((sum, v) => sum + v.montant_total, 0);
            return {
                name: site.nom,
                value: total,
                color: generateSiteColor(site.id)
            };
        }).filter(d => d.value > 0);
        
        // Détruire le graphique existant
        if (sitesChart) {
            sitesChart.destroy();
        }
        
        // Créer le graphique en camembert
        sitesChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: siteData.map(d => d.name),
                datasets: [{
                    data: siteData.map(d => d.value),
                    backgroundColor: siteData.map(d => d.color),
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed / total) * 100).toFixed(1);
                                return `${context.label}: ${formatCurrency(context.parsed)} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
        
    } catch (error) {
        console.error('Erreur création graphique sites:', error);
    }
}

// Génération de couleurs pour les sites
function generateSiteColor(siteId) {
    const colors = [
        '#2196F3', '#4CAF50', '#FF9800', '#F44336',
        '#9C27B0', '#607D8B', '#795548', '#E91E63'
    ];
    return colors[siteId % colors.length];
}

// Chargement des transactions récentes
async function loadRecentTransactions() {
    try {
        const transactions = await dbManager.readAll('transactions');
        const sites = await dbManager.readAll('sites');
        
        // Prendre les 10 dernières transactions
        const recentTransactions = transactions
            .sort((a, b) => new Date(b.date_creation) - new Date(a.date_creation))
            .slice(0, 10);
        
        const tableBody = document.querySelector('#recent-transactions tbody');
        if (!tableBody) return;
        
        tableBody.innerHTML = '';
        
        recentTransactions.forEach(transaction => {
            const site = sites.find(s => s.id === transaction.site_id);
            const row = createTransactionRow(transaction, site);
            tableBody.appendChild(row);
        });
        
    } catch (error) {
        console.error('Erreur chargement transactions récentes:', error);
    }
}

function createTransactionRow(transaction, site) {
    const row = document.createElement('tr');
    row.className = 'transaction-row';
    
    const statusClass = getTransactionStatusClass(transaction.categorie);
    const typeIcon = getTransactionTypeIcon(transaction.type_transaction);
    
    row.innerHTML = `
        <td>${formatDate(transaction.date_transaction)}</td>
        <td>${site ? site.nom : 'Site inconnu'}</td>
        <td>
            <i class="fas ${typeIcon}"></i>
            ${transaction.categorie}
        </td>
        <td class="${transaction.type_transaction === 'sortie' ? 'negative' : 'positive'}">
            ${transaction.type_transaction === 'sortie' ? '-' : '+'}${formatCurrency(transaction.montant)}
        </td>
        <td>
            <span class="badge ${statusClass}">
                ${getTransactionStatusText(transaction.categorie)}
            </span>
        </td>
    `;
    
    // Ajouter un effet hover
    row.addEventListener('mouseenter', () => {
        row.style.backgroundColor = 'var(--background-dark)';
    });
    
    row.addEventListener('mouseleave', () => {
        row.style.backgroundColor = '';
    });
    
    return row;
}

function getTransactionStatusClass(categorie) {
    const statusClasses = {
        'achat': 'badge-warning',
        'vente': 'badge-success',
        'frais': 'badge-danger',
        'ajustement': 'badge-secondary'
    };
    return statusClasses[categorie] || 'badge-secondary';
}

function getTransactionTypeIcon(type) {
    return type === 'entree' ? 'fa-arrow-down' : 'fa-arrow-up';
}

function getTransactionStatusText(categorie) {
    const statusTexts = {
        'achat': 'Achat',
        'vente': 'Vente',
        'frais': 'Frais',
        'ajustement': 'Ajustement'
    };
    return statusTexts[categorie] || categorie;
}

// Affichage du loading
function showDashboardLoading(show) {
    const elements = [
        '#total-budget',
        '#total-achats', 
        '#total-ventes',
        '#roi-global'
    ];
    
    elements.forEach(selector => {
        const element = document.querySelector(selector);
        if (element) {
            if (show) {
                element.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            }
        }
    });
    
    // Loading pour les graphiques
    const chartsContainers = [
        '#sales-chart',
        '#sites-chart'
    ];
    
    chartsContainers.forEach(selector => {
        const container = document.querySelector(selector);
        if (container) {
            const parent = container.closest('.chart-content');
            if (show) {
                parent.classList.add('loading');
            } else {
                parent.classList.remove('loading');
            }
        }
    });
}

// Actualisation automatique du dashboard
function setupDashboardAutoRefresh() {
    // Actualiser toutes les 5 minutes
    setInterval(() => {
        if (document.getElementById('page-dashboard').classList.contains('active')) {
            loadDashboardData();
        }
    }, 5 * 60 * 1000);
}

// Initialisation du dashboard
document.addEventListener('DOMContentLoaded', () => {
    setupDashboardAutoRefresh();
});

// Export des fonctions
window.loadDashboardData = loadDashboardData;