// js/app.js - JavaScript principal pour PWA Gestion Financement

// Variables globales
let currentUser = null;
let currentSite = null;
let isOnline = navigator.onLine;

// Initialisation de l'application
async function initApp() {
    try {
        console.log('Initialisation de l\'application...');
        
        // Vérifier la session utilisateur
        await loadUserSession();
        
        // Initialiser les composants UI
        initializeUI();
        
        // Configurer les événements
        setupEventListeners();
        
        // Charger les données initiales
        await loadInitialData();
        
        // Configurer la synchronisation
        setupSync();
        
        // Démarrer sur le dashboard
        showPage('dashboard');
        
        console.log('Application initialisée avec succès');
        
    } catch (error) {
        console.error('Erreur lors de l\'initialisation:', error);
        showError('Erreur lors de l\'initialisation de l\'application');
    }
}

// Gestion de la session utilisateur
async function loadUserSession() {
    try {
        // Vérifier s'il y a un utilisateur connecté
        const savedUser = localStorage.getItem('currentUser');
        if (savedUser) {
            currentUser = JSON.parse(savedUser);
            updateUserInterface();
        } else {
            // Rediriger vers la page de connexion si pas d'utilisateur
            showLoginForm();
        }
    } catch (error) {
        console.error('Erreur chargement session:', error);
        showLoginForm();
    }
}

function updateUserInterface() {
    if (currentUser) {
        document.getElementById('user-name').textContent = currentUser.nom;
        document.getElementById('app').style.display = 'block';
    }
}

// Initialisation de l'interface utilisateur
function initializeUI() {
    // Configuration du menu mobile
    const menuToggle = document.getElementById('menu-toggle');
    const sidebar = document.getElementById('sidebar');
    
    menuToggle?.addEventListener('click', () => {
        sidebar.classList.toggle('show');
    });
    
    // Fermer le menu en cliquant à l'extérieur
    document.addEventListener('click', (e) => {
        if (!sidebar.contains(e.target) && !menuToggle.contains(e.target)) {
            sidebar.classList.remove('show');
        }
    });
    
    // Configuration du menu utilisateur
    const userButton = document.getElementById('user-button');
    const userDropdown = document.getElementById('user-dropdown');
    
    userButton?.addEventListener('click', (e) => {
        e.stopPropagation();
        userDropdown.classList.toggle('show');
    });
    
    document.addEventListener('click', () => {
        userDropdown.classList.remove('show');
    });
    
    // Mise à jour du statut de connexion
    updateConnectionStatus();
}

// Configuration des événements
function setupEventListeners() {
    // Événements de connectivité
    window.addEventListener('online', () => {
        isOnline = true;
        updateConnectionStatus();
        syncManager?.syncAll();
        showNotification('Connexion rétablie', 'success');
    });
    
    window.addEventListener('offline', () => {
        isOnline = false;
        updateConnectionStatus();
        showNotification('Mode hors ligne activé', 'warning');
    });
    
    // Événements de formulaires
    setupFormHandlers();
    
    // Raccourcis clavier
    document.addEventListener('keydown', handleKeyboardShortcuts);
    
    // Gestion des liens de navigation
    document.querySelectorAll('.nav-link').forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const page = link.getAttribute('onclick')?.match(/showPage\('([^']+)'\)/)?.[1];
            if (page) {
                showPage(page);
                // Fermer le sidebar sur mobile
                document.getElementById('sidebar').classList.remove('show');
            }
        });
    });
}

// Gestion des formulaires
function setupFormHandlers() {
    // Formulaire de site
    const formSite = document.getElementById('form-site');
    formSite?.addEventListener('submit', async (e) => {
        e.preventDefault();
        await handleSiteSubmit(e);
    });
    
    // Auto-save pour les formulaires
    document.querySelectorAll('form[data-autosave]').forEach(form => {
        form.addEventListener('input', debounce(() => {
            saveFormData(form);
        }, 1000));
    });
}

// Chargement des données initiales
async function loadInitialData() {
    try {
        // Charger les sites
        await loadSites();
        
        // Charger les KPI du dashboard
        await loadDashboardData();
        
        // Charger les données de base (produits, fournisseurs, etc.)
        await loadMasterData();
        
    } catch (error) {
        console.error('Erreur chargement données initiales:', error);
    }
}

// Navigation entre les pages
function showPage(pageId) {
    // Masquer toutes les pages
    document.querySelectorAll('.page').forEach(page => {
        page.classList.remove('active');
    });
    
    // Mettre à jour la navigation
    document.querySelectorAll('.nav-link').forEach(link => {
        link.classList.remove('active');
    });
    
    // Afficher la page demandée
    const targetPage = document.getElementById(`page-${pageId}`);
    if (targetPage) {
        targetPage.classList.add('active');
        
        // Activer le lien de navigation correspondant
        const navLink = document.querySelector(`[onclick*="showPage('${pageId}')"]`);
        navLink?.classList.add('active');
        
        // Charger les données spécifiques à la page
        loadPageData(pageId);
        
        // Mettre à jour l'URL sans recharger
        updateURLState(pageId);
    }
}

// Chargement des données spécifiques à chaque page
async function loadPageData(pageId) {
    try {
        switch (pageId) {
            case 'dashboard':
                await loadDashboardData();
                break;
            case 'sites':
                await loadSitesData();
                break;
            case 'achats':
                await loadAchatsData();
                break;
            case 'stocks':
                await loadStocksData();
                break;
            case 'ventes':
                await loadVentesData();
                break;
            case 'reporting':
                await loadReportingData();
                break;
            case 'fournisseurs':
                await loadFournisseursData();
                break;
            case 'clients':
                await loadClientsData();
                break;
            case 'admin':
                await loadAdminData();
                break;
        }
    } catch (error) {
        console.error(`Erreur chargement page ${pageId}:`, error);
        showError(`Erreur lors du chargement de la page ${pageId}`);
    }
}

// Mise à jour du statut de connexion
function updateConnectionStatus() {
    const connectionIcon = document.getElementById('connection-icon');
    const connectionText = document.getElementById('connection-text');
    const syncStatus = document.getElementById('sync-status');
    
    if (isOnline) {
        connectionIcon.className = 'fas fa-wifi';
        connectionText.textContent = 'En ligne';
        syncStatus.className = 'sync-status online';
    } else {
        connectionIcon.className = 'fas fa-wifi-slash';
        connectionText.textContent = 'Hors ligne';
        syncStatus.className = 'sync-status offline';
    }
}

// Gestion des modals
function openModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('show');
        document.body.style.overflow = 'hidden';
        
        // Focus sur le premier champ
        const firstInput = modal.querySelector('input, select, textarea');
        firstInput?.focus();
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('show');
        document.body.style.overflow = '';
        
        // Réinitialiser le formulaire
        const form = modal.querySelector('form');
        form?.reset();
        clearFormErrors(form);
    }
}

// Fermer modal avec échap
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        document.querySelectorAll('.modal.show').forEach(modal => {
            modal.classList.remove('show');
        });
        document.body.style.overflow = '';
    }
});

// Gestion des notifications
function showNotification(message, type = 'info', duration = 3000) {
    const notification = createNotificationElement(message, type);
    document.body.appendChild(notification);
    
    // Afficher avec animation
    setTimeout(() => notification.classList.add('show'), 10);
    
    // Masquer automatiquement
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, duration);
}

function createNotificationElement(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas ${getNotificationIcon(type)}"></i>
            <span>${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    return notification;
}

function getNotificationIcon(type) {
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    return icons[type] || icons.info;
}

// Gestion des erreurs
function showError(message, details = null) {
    console.error('Erreur:', message, details);
    showNotification(message, 'error', 5000);
}

// Validation des formulaires
function validateForm(form) {
    const errors = [];
    const requiredFields = form.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            errors.push({
                field: field.name,
                message: `Le champ ${field.labels?.[0]?.textContent || field.name} est requis`
            });
        }
    });
    
    // Validation email
    const emailFields = form.querySelectorAll('input[type="email"]');
    emailFields.forEach(field => {
        if (field.value && !isValidEmail(field.value)) {
            errors.push({
                field: field.name,
                message: 'Format d\'email invalide'
            });
        }
    });
    
    // Validation numérique
    const numberFields = form.querySelectorAll('input[type="number"]');
    numberFields.forEach(field => {
        if (field.value && isNaN(field.value)) {
            errors.push({
                field: field.name,
                message: 'Valeur numérique invalide'
            });
        }
    });
    
    return errors;
}

function displayFormErrors(form, errors) {
    // Effacer les erreurs précédentes
    clearFormErrors(form);
    
    errors.forEach(error => {
        const field = form.querySelector(`[name="${error.field}"]`);
        if (field) {
            field.classList.add('error');
            
            const errorElement = document.createElement('div');
            errorElement.className = 'form-error';
            errorElement.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${error.message}`;
            
            field.parentNode.appendChild(errorElement);
        }
    });
}

function clearFormErrors(form) {
    form?.querySelectorAll('.form-error').forEach(error => error.remove());
    form?.querySelectorAll('.error').forEach(field => field.classList.remove('error'));
}

// Utilitaires
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function formatCurrency(amount, currency = 'XOF') {
    return new Intl.NumberFormat('fr-FR', {
        style: 'currency',
        currency: currency,
        minimumFractionDigits: 0
    }).format(amount);
}

function formatDate(date, format = 'short') {
    const options = {
        short: { day: '2-digit', month: '2-digit', year: 'numeric' },
        long: { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' },
        time: { hour: '2-digit', minute: '2-digit' }
    };
    
    return new Intl.DateTimeFormat('fr-FR', options[format]).format(new Date(date));
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Sauvegarde automatique des formulaires
function saveFormData(form) {
    const formData = new FormData(form);
    const data = Object.fromEntries(formData.entries());
    const formId = form.id || 'unnamed_form';
    
    localStorage.setItem(`form_${formId}`, JSON.stringify(data));
}

function loadFormData(form) {
    const formId = form.id || 'unnamed_form';
    const savedData = localStorage.getItem(`form_${formId}`);
    
    if (savedData) {
        const data = JSON.parse(savedData);
        Object.keys(data).forEach(key => {
            const field = form.querySelector(`[name="${key}"]`);
            if (field) {
                field.value = data[key];
            }
        });
    }
}

function clearFormData(formId) {
    localStorage.removeItem(`form_${formId}`);
}

// Raccourcis clavier
function handleKeyboardShortcuts(e) {
    // Ctrl/Cmd + S pour sauvegarder
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        const activeForm = document.querySelector('form:focus-within');
        if (activeForm) {
            activeForm.dispatchEvent(new Event('submit'));
        }
    }
    
    // Échap pour fermer les modals
    if (e.key === 'Escape') {
        const openModal = document.querySelector('.modal.show');
        if (openModal) {
            openModal.classList.remove('show');
            document.body.style.overflow = '';
        }
    }
    
    // Ctrl/Cmd + K pour recherche
    if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
        e.preventDefault();
        const searchInput = document.querySelector('input[type="search"]');
        searchInput?.focus();
    }
}

// Gestion de l'état de l'URL
function updateURLState(pageId) {
    const url = new URL(window.location);
    url.searchParams.set('page', pageId);
    window.history.pushState({ page: pageId }, '', url);
}

// Gestion du bouton retour
window.addEventListener('popstate', (e) => {
    const page = e.state?.page || 'dashboard';
    showPage(page);
});

// Synchronisation des données
function syncData() {
    if (!isOnline) {
        showNotification('Synchronisation impossible en mode hors ligne', 'warning');
        return;
    }
    
    showNotification('Synchronisation en cours...', 'info');
    const syncIcon = document.querySelector('#sync-status i');
    syncIcon.className = 'fas fa-sync fa-spin';
    
    syncManager?.syncAll()
        .then(() => {
            showNotification('Synchronisation terminée', 'success');
        })
        .catch((error) => {
            console.error('Erreur synchronisation:', error);
            showNotification('Erreur lors de la synchronisation', 'error');
        })
        .finally(() => {
            syncIcon.className = 'fas fa-wifi';
        });
}

// Déconnexion
function logout() {
    localStorage.removeItem('currentUser');
    localStorage.removeItem('authToken');
    currentUser = null;
    
    // Effacer les données sensibles
    if (dbManager) {
        // Optionnel: effacer la base locale pour sécurité
        // indexedDB.deleteDatabase(dbManager.dbName);
    }
    
    showNotification('Déconnexion réussie', 'success');
    
    // Rediriger vers la page de connexion
    setTimeout(() => {
        window.location.href = '/login.html';
    }, 1000);
}

// Affichage du profil utilisateur
function showProfile() {
    // À implémenter: modal de profil utilisateur
    console.log('Affichage du profil utilisateur');
}

// Masquer le splash screen
function hideSplashScreen() {
    const splashScreen = document.getElementById('splash-screen');
    if (splashScreen) {
        splashScreen.style.opacity = '0';
        setTimeout(() => {
            splashScreen.style.display = 'none';
            document.getElementById('app').style.display = 'grid';
        }, 500);
    }
}

// Formulaire de connexion simple (à remplacer par une vraie page)
function showLoginForm() {
    // Simuler une connexion pour le développement
    currentUser = {
        id: 1,
        nom: 'Administrateur',
        email: 'admin@example.com',
        role: 'admin'
    };
    
    localStorage.setItem('currentUser', JSON.stringify(currentUser));
    updateUserInterface();
}

// Chargement des données maîtres
async function loadMasterData() {
    try {
        // Charger les produits s'ils n'existent pas
        const produits = await dbManager.readAll('produits');
        if (produits.length === 0) {
            await initializeDefaultProducts();
        }
        
        // Charger les autres données de base...
        
    } catch (error) {
        console.error('Erreur chargement données maîtres:', error);
    }
}

// Initialiser les produits par défaut
async function initializeDefaultProducts() {
    const defaultProducts = [
        { nom: 'Maïs', unite_mesure: 'kg', categorie: 'cereale' },
        { nom: 'Riz', unite_mesure: 'kg', categorie: 'cereale' },
        { nom: 'Mil', unite_mesure: 'kg', categorie: 'cereale' },
        { nom: 'Sorgho', unite_mesure: 'kg', categorie: 'cereale' },
        { nom: 'Haricot', unite_mesure: 'kg', categorie: 'legumineuse' },
        { nom: 'Niébé', unite_mesure: 'kg', categorie: 'legumineuse' },
        { nom: 'Igname', unite_mesure: 'kg', categorie: 'tubercule' },
        { nom: 'Manioc', unite_mesure: 'kg', categorie: 'tubercule' }
    ];
    
    for (const product of defaultProducts) {
        await dbManager.create('produits', product);
    }
}

// Configuration de la synchronisation
function setupSync() {
    if (syncManager) {
        // Synchronisation périodique toutes les 5 minutes si en ligne
        setInterval(() => {
            if (isOnline) {
                syncManager.syncAll();
            }
        }, 5 * 60 * 1000);
    }
}

// Export des fonctions globales
window.showPage = showPage;
window.openModal = openModal;
window.closeModal = closeModal;
window.syncData = syncData;
window.logout = logout;
window.showProfile = showProfile;