// js/achats.js - Gestion des achats

// Variables globales pour les achats
let currentAchats = [];
let selectedAchat = null;
let fournisseurs = [];
let produits = [];

// Chargement des données des achats
async function loadAchatsData() {
    try {
        showAchatsLoading(true);
        
        // Charger les données nécessaires
        await Promise.all([
            loadAchats(),
            loadFournisseurs(),
            loadProduits()
        ]);
        
        // Afficher les achats
        displayAchatsTable(currentAchats);
        
        // Charger les statistiques
        await loadAchatsStatistics();
        
        showAchatsLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement achats:', error);
        showError('Erreur lors du chargement des achats');
        showAchatsLoading(false);
    }
}

// Chargement de la liste des achats
async function loadAchats() {
    try {
        const response = await fetch('/api/achats', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (!response.ok) {
            throw new Error('Erreur de chargement des achats');
        }
        
        const data = await response.json();
        currentAchats = data.data || [];
        
    } catch (error) {
        console.error('Erreur chargement achats:', error);
        // Fallback vers la base locale
        currentAchats = await dbManager.readAll('achats') || [];
    }
}

// Chargement des fournisseurs
async function loadFournisseurs() {
    try {
        const response = await fetch('/api/fournisseurs', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            fournisseurs = data.data || [];
        } else {
            // Fallback vers la base locale
            fournisseurs = await dbManager.readAll('fournisseurs') || [];
        }
        
    } catch (error) {
        console.error('Erreur chargement fournisseurs:', error);
        fournisseurs = await dbManager.readAll('fournisseurs') || [];
    }
}

// Chargement des produits
async function loadProduits() {
    try {
        const response = await fetch('/api/produits', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            produits = data.data || [];
        } else {
            // Fallback vers la base locale
            produits = await dbManager.readAll('produits') || [];
        }
        
    } catch (error) {
        console.error('Erreur chargement produits:', error);
        produits = await dbManager.readAll('produits') || [];
    }
}

// Affichage du tableau des achats
function displayAchatsTable(achats) {
    const tableContainer = document.querySelector('#achats-table-container');
    if (!tableContainer) return;
    
    if (achats.length === 0) {
        tableContainer.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-shopping-cart"></i>
                <h3>Aucun achat enregistré</h3>
                <p>Commencez par enregistrer votre premier achat</p>
                <button class="btn btn-primary" onclick="openModal('modal-achat')">
                    <i class="fas fa-plus"></i> Nouvel achat
                </button>
            </div>
        `;
        return;
    }
    
    const table = `
        <div class="table-card">
            <div class="table-header">
                <h3>Liste des achats</h3>
                <div class="table-actions">
                    <div class="search-box">
                        <input type="search" placeholder="Rechercher..." id="achats-search">
                        <i class="fas fa-search"></i>
                    </div>
                    <select id="achats-filter-site">
                        <option value="">Tous les sites</option>
                        ${getSitesOptions()}
                    </select>
                    <select id="achats-filter-statut">
                        <option value="">Tous les statuts</option>
                        <option value="en_cours">En cours</option>
                        <option value="livre">Livré</option>
                        <option value="paye">Payé</option>
                    </select>
                </div>
            </div>
            <div class="table-content">
                <table id="achats-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Site</th>
                            <th>Fournisseur</th>
                            <th>Produit</th>
                            <th>Quantité</th>
                            <th>Prix unitaire</th>
                            <th>Coût total</th>
                            <th>Statut</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${achats.map(achat => createAchatRow(achat)).join('')}
                    </tbody>
                </table>
            </div>
        </div>
    `;
    
    tableContainer.innerHTML = table;
    
    // Ajouter les événements de filtrage
    setupAchatsFilters();
}

// Création d'une ligne de tableau pour un achat
function createAchatRow(achat) {
    const site = achat.site ? achat.site.nom : 'Site inconnu';
    const fournisseur = achat.fournisseur ? achat.fournisseur.nom : 'Fournisseur inconnu';
    const produit = achat.produit ? achat.produit.nom : 'Produit inconnu';
    const unite = achat.produit ? achat.produit.unite_mesure : 'kg';
    
    return `
        <tr data-achat-id="${achat.id}">
            <td>${formatDate(achat.date_achat)}</td>
            <td>${site}</td>
            <td>${fournisseur}</td>
            <td>${produit}</td>
            <td>${achat.quantite} ${unite}</td>
            <td>${formatCurrency(achat.prix_unitaire)}</td>
            <td>${formatCurrency(achat.cout_total)}</td>
            <td>
                <span class="badge ${getAchatStatusClass(achat.statut)}">
                    ${getAchatStatusText(achat.statut)}
                </span>
            </td>
            <td>
                <div class="actions">
                    <button class="btn btn-sm btn-outline" onclick="viewAchat(${achat.id})" title="Voir détails">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-primary" onclick="editAchat(${achat.id})" title="Modifier">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteAchat(${achat.id})" title="Supprimer">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `;
}

// Configuration des filtres
function setupAchatsFilters() {
    const searchInput = document.getElementById('achats-search');
    const siteFilter = document.getElementById('achats-filter-site');
    const statutFilter = document.getElementById('achats-filter-statut');
    
    [searchInput, siteFilter, statutFilter].forEach(element => {
        if (element) {
            element.addEventListener('input', filterAchats);
        }
    });
}

// Filtrage des achats
function filterAchats() {
    const searchTerm = document.getElementById('achats-search')?.value.toLowerCase() || '';
    const siteFilter = document.getElementById('achats-filter-site')?.value || '';
    const statutFilter = document.getElementById('achats-filter-statut')?.value || '';
    
    const filteredAchats = currentAchats.filter(achat => {
        const matchSearch = !searchTerm || 
            (achat.fournisseur?.nom.toLowerCase().includes(searchTerm)) ||
            (achat.produit?.nom.toLowerCase().includes(searchTerm)) ||
            (achat.numero_bon?.toLowerCase().includes(searchTerm));
            
        const matchSite = !siteFilter || achat.site_id == siteFilter;
        const matchStatut = !statutFilter || achat.statut === statutFilter;
        
        return matchSearch && matchSite && matchStatut;
    });
    
    // Re-afficher uniquement le tbody
    const tbody = document.querySelector('#achats-table tbody');
    if (tbody) {
        tbody.innerHTML = filteredAchats.map(achat => createAchatRow(achat)).join('');
    }
}

// Gestion du formulaire d'achat
async function handleAchatSubmit(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    
    // Validation
    const errors = validateForm(form);
    if (errors.length > 0) {
        displayFormErrors(form, errors);
        return;
    }
    
    try {
        const achatData = {
            site_id: parseInt(formData.get('site_id')),
            fournisseur_id: parseInt(formData.get('fournisseur_id')),
            produit_id: parseInt(formData.get('produit_id')),
            quantite: parseFloat(formData.get('quantite')),
            prix_unitaire: parseFloat(formData.get('prix_unitaire')),
            frais_transport: parseFloat(formData.get('frais_transport')) || 0,
            autres_frais: parseFloat(formData.get('autres_frais')) || 0,
            date_achat: formData.get('date_achat'),
            numero_bon: formData.get('numero_bon'),
            statut: formData.get('statut') || 'en_cours'
        };
        
        // Calculs automatiques
        achatData.montant_total = achatData.quantite * achatData.prix_unitaire;
        achatData.cout_total = achatData.montant_total + achatData.frais_transport + achatData.autres_frais;
        
        if (isOnline) {
            // Envoyer vers l'API
            const response = await fetch('/api/achats', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                },
                body: JSON.stringify(achatData)
            });
            
            if (response.ok) {
                showNotification('Achat enregistré avec succès', 'success');
            } else {
                throw new Error('Erreur lors de l\'enregistrement');
            }
        } else {
            // Enregistrer localement
            await dbManager.create('achats', achatData);
            showNotification('Achat enregistré localement (sera synchronisé)', 'info');
        }
        
        closeModal('modal-achat');
        await loadAchatsData();
        
    } catch (error) {
        console.error('Erreur création achat:', error);
        showError('Erreur lors de l\'enregistrement de l\'achat');
    }
}

// Calcul automatique des totaux
function calculateAchatTotals() {
    const quantite = parseFloat(document.getElementById('achat-quantite')?.value) || 0;
    const prixUnitaire = parseFloat(document.getElementById('achat-prix-unitaire')?.value) || 0;
    const fraisTransport = parseFloat(document.getElementById('achat-frais-transport')?.value) || 0;
    const autresFrais = parseFloat(document.getElementById('achat-autres-frais')?.value) || 0;
    
    const montantTotal = quantite * prixUnitaire;
    const coutTotal = montantTotal + fraisTransport + autresFrais;
    
    // Mettre à jour l'affichage
    const montantElement = document.getElementById('achat-montant-total');
    const coutElement = document.getElementById('achat-cout-total');
    
    if (montantElement) montantElement.textContent = formatCurrency(montantTotal);
    if (coutElement) coutElement.textContent = formatCurrency(coutTotal);
}

// Utilitaires pour les achats
function getAchatStatusClass(statut) {
    const classes = {
        'en_cours': 'badge-warning',
        'livre': 'badge-info',
        'paye': 'badge-success'
    };
    return classes[statut] || 'badge-secondary';
}

function getAchatStatusText(statut) {
    const texts = {
        'en_cours': 'En cours',
        'livre': 'Livré',
        'paye': 'Payé'
    };
    return texts[statut] || statut;
}

function getSitesOptions() {
    // Récupérer les sites depuis currentSites ou la base locale
    const sites = currentSites || [];
    return sites.map(site => `<option value="${site.id}">${site.nom}</option>`).join('');
}

// Actions sur les achats
function viewAchat(id) {
    const achat = currentAchats.find(a => a.id === id);
    if (achat) {
        // Afficher les détails de l'achat
        showAchatDetails(achat);
    }
}

function editAchat(id) {
    const achat = currentAchats.find(a => a.id === id);
    if (achat) {
        // Pré-remplir le formulaire et ouvrir le modal
        populateAchatForm(achat);
        openModal('modal-achat');
    }
}

async function deleteAchat(id) {
    if (!confirm('Êtes-vous sûr de vouloir supprimer cet achat ?')) {
        return;
    }
    
    try {
        if (isOnline) {
            const response = await fetch(`/api/achats/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                }
            });
            
            if (response.ok) {
                showNotification('Achat supprimé avec succès', 'success');
            } else {
                throw new Error('Erreur lors de la suppression');
            }
        } else {
            await dbManager.delete('achats', id);
            showNotification('Achat supprimé localement', 'info');
        }
        
        await loadAchatsData();
        
    } catch (error) {
        console.error('Erreur suppression achat:', error);
        showError('Erreur lors de la suppression de l\'achat');
    }
}

// Chargement des statistiques des achats
async function loadAchatsStatistics() {
    try {
        const currentMonth = new Date().toISOString().substring(0, 7);
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        const achatsMonth = currentAchats.filter(a => 
            a.date_achat >= startOfMonth && a.date_achat <= endOfMonth
        );
        
        const stats = {
            total_achats: achatsMonth.reduce((sum, a) => sum + a.cout_total, 0),
            nb_achats: achatsMonth.length,
            quantite_totale: achatsMonth.reduce((sum, a) => sum + a.quantite, 0),
            prix_moyen: achatsMonth.length > 0 ? 
                achatsMonth.reduce((sum, a) => sum + a.prix_unitaire, 0) / achatsMonth.length : 0
        };
        
        displayAchatsStatistics(stats);
        
    } catch (error) {
        console.error('Erreur calcul statistiques achats:', error);
    }
}

function displayAchatsStatistics(stats) {
    const statsContainer = document.querySelector('.achats-statistics');
    if (statsContainer) {
        statsContainer.innerHTML = `
            <div class="stat-card">
                <div class="stat-value">${formatCurrency(stats.total_achats)}</div>
                <div class="stat-label">Total achats ce mois</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${stats.nb_achats}</div>
                <div class="stat-label">Nombre d'achats</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${stats.quantite_totale.toFixed(0)}</div>
                <div class="stat-label">Quantité totale</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${formatCurrency(stats.prix_moyen)}</div>
                <div class="stat-label">Prix moyen</div>
            </div>
        `;
    }
}

function showAchatsLoading(show) {
    const tableContainer = document.querySelector('#achats-table-container');
    if (!tableContainer) return;
    
    if (show) {
        tableContainer.innerHTML = `
            <div class="loading-container">
                <div class="loading-spinner"></div>
                <p>Chargement des achats...</p>
            </div>
        `;
    }
}

// Export des fonctions
window.loadAchatsData = loadAchatsData;
window.handleAchatSubmit = handleAchatSubmit;
window.calculateAchatTotals = calculateAchatTotals;
window.viewAchat = viewAchat;
window.editAchat = editAchat;
window.deleteAchat = deleteAchat;

//////////////////////////////////////////////
// js/ventes.js - Gestion des ventes
//////////////////////////////////////////////

// Variables globales pour les ventes
let currentVentes = [];
let selectedVente = null;
let clients = [];

// Chargement des données des ventes
async function loadVentesData() {
    try {
        showVentesLoading(true);
        
        // Charger les données nécessaires
        await Promise.all([
            loadVentes(),
            loadClients(),
            loadProduits(), // Réutilise la fonction des achats
            loadSites()
        ]);
        
        // Afficher les ventes
        displayVentesTable(currentVentes);
        
        // Charger les statistiques
        await loadVentesStatistics();
        
        showVentesLoading(false);
        
    } catch (error) {
        console.error('Erreur chargement ventes:', error);
        showError('Erreur lors du chargement des ventes');
        showVentesLoading(false);
    }
}

// Chargement de la liste des ventes
async function loadVentes() {
    try {
        const response = await fetch('/api/ventes', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (!response.ok) {
            throw new Error('Erreur de chargement des ventes');
        }
        
        const data = await response.json();
        currentVentes = data.data || [];
        
    } catch (error) {
        console.error('Erreur chargement ventes:', error);
        // Fallback vers la base locale
        currentVentes = await dbManager.readAll('ventes') || [];
    }
}

// Chargement des clients
async function loadClients() {
    try {
        const response = await fetch('/api/clients', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            clients = data.data || [];
        } else {
            // Fallback vers la base locale
            clients = await dbManager.readAll('clients') || [];
        }
        
    } catch (error) {
        console.error('Erreur chargement clients:', error);
        clients = await dbManager.readAll('clients') || [];
    }
}

// Affichage du tableau des ventes
function displayVentesTable(ventes) {
    const tableContainer = document.querySelector('#ventes-table-container');
    if (!tableContainer) return;
    
    if (ventes.length === 0) {
        tableContainer.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-cash-register"></i>
                <h3>Aucune vente enregistrée</h3>
                <p>Commencez par enregistrer votre première vente</p>
                <button class="btn btn-primary" onclick="openModal('modal-vente')">
                    <i class="fas fa-plus"></i> Nouvelle vente
                </button>
            </div>
        `;
        return;
    }
    
    const table = `
        <div class="table-card">
            <div class="table-header">
                <h3>Liste des ventes</h3>
                <div class="table-actions">
                    <div class="search-box">
                        <input type="search" placeholder="Rechercher..." id="ventes-search">
                        <i class="fas fa-search"></i>
                    </div>
                    <select id="ventes-filter-site">
                        <option value="">Tous les sites</option>
                        ${getSitesOptions()}
                    </select>
                    <select id="ventes-filter-statut">
                        <option value="">Tous les statuts</option>
                        <option value="en_cours">En cours</option>
                        <option value="livre">Livré</option>
                        <option value="paye">Payé</option>
                    </select>
                </div>
            </div>
            <div class="table-content">
                <table id="ventes-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Site</th>
                            <th>Client</th>
                            <th>Produit</th>
                            <th>Quantité</th>
                            <th>Prix unitaire</th>
                            <th>Montant total</th>
                            <th>Marge</th>
                            <th>Statut</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${ventes.map(vente => createVenteRow(vente)).join('')}
                    </tbody>
                </table>
            </div>
        </div>
    `;
    
    tableContainer.innerHTML = table;
    
    // Ajouter les événements de filtrage
    setupVentesFilters();
}

// Création d'une ligne de tableau pour une vente
function createVenteRow(vente) {
    const site = vente.site ? vente.site.nom : 'Site inconnu';
    const client = vente.client ? vente.client.nom : 'Client inconnu';
    const produit = vente.produit ? vente.produit.nom : 'Produit inconnu';
    const unite = vente.produit ? vente.produit.unite_mesure : 'kg';
    
    const margeClass = vente.marge_brute > 0 ? 'positive' : (vente.marge_brute < 0 ? 'negative' : '');
    
    return `
        <tr data-vente-id="${vente.id}">
            <td>${formatDate(vente.date_vente)}</td>
            <td>${site}</td>
            <td>${client}</td>
            <td>${produit}</td>
            <td>${vente.quantite} ${unite}</td>
            <td>${formatCurrency(vente.prix_unitaire)}</td>
            <td>${formatCurrency(vente.montant_total)}</td>
            <td class="${margeClass}">${formatCurrency(vente.marge_brute)}</td>
            <td>
                <span class="badge ${getVenteStatusClass(vente.statut)}">
                    ${getVenteStatusText(vente.statut)}
                </span>
            </td>
            <td>
                <div class="actions">
                    <button class="btn btn-sm btn-outline" onclick="viewVente(${vente.id})" title="Voir détails">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-primary" onclick="editVente(${vente.id})" title="Modifier">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteVente(${vente.id})" title="Supprimer">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `;
}

// Configuration des filtres pour les ventes
function setupVentesFilters() {
    const searchInput = document.getElementById('ventes-search');
    const siteFilter = document.getElementById('ventes-filter-site');
    const statutFilter = document.getElementById('ventes-filter-statut');
    
    [searchInput, siteFilter, statutFilter].forEach(element => {
        if (element) {
            element.addEventListener('input', filterVentes);
        }
    });
}

// Filtrage des ventes
function filterVentes() {
    const searchTerm = document.getElementById('ventes-search')?.value.toLowerCase() || '';
    const siteFilter = document.getElementById('ventes-filter-site')?.value || '';
    const statutFilter = document.getElementById('ventes-filter-statut')?.value || '';
    
    const filteredVentes = currentVentes.filter(vente => {
        const matchSearch = !searchTerm || 
            (vente.client?.nom.toLowerCase().includes(searchTerm)) ||
            (vente.produit?.nom.toLowerCase().includes(searchTerm)) ||
            (vente.numero_facture?.toLowerCase().includes(searchTerm));
            
        const matchSite = !siteFilter || vente.site_id == siteFilter;
        const matchStatut = !statutFilter || vente.statut === statutFilter;
        
        return matchSearch && matchSite && matchStatut;
    });
    
    // Re-afficher uniquement le tbody
    const tbody = document.querySelector('#ventes-table tbody');
    if (tbody) {
        tbody.innerHTML = filteredVentes.map(vente => createVenteRow(vente)).join('');
    }
}

// Gestion du formulaire de vente
async function handleVenteSubmit(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    
    // Validation
    const errors = validateForm(form);
    if (errors.length > 0) {
        displayFormErrors(form, errors);
        return;
    }
    
    try {
        const venteData = {
            site_id: parseInt(formData.get('site_id')),
            client_id: parseInt(formData.get('client_id')),
            produit_id: parseInt(formData.get('produit_id')),
            quantite: parseFloat(formData.get('quantite')),
            prix_unitaire: parseFloat(formData.get('prix_unitaire')),
            date_vente: formData.get('date_vente'),
            numero_facture: formData.get('numero_facture'),
            statut: formData.get('statut') || 'en_cours'
        };
        
        // Calculs automatiques
        venteData.montant_total = venteData.quantite * venteData.prix_unitaire;
        
        if (isOnline) {
            // Envoyer vers l'API
            const response = await fetch('/api/ventes', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                },
                body: JSON.stringify(venteData)
            });
            
            if (response.ok) {
                showNotification('Vente enregistrée avec succès', 'success');
            } else {
                throw new Error('Erreur lors de l\'enregistrement');
            }
        } else {
            // Enregistrer localement
            await dbManager.create('ventes', venteData);
            showNotification('Vente enregistrée localement (sera synchronisée)', 'info');
        }
        
        closeModal('modal-vente');
        await loadVentesData();
        
    } catch (error) {
        console.error('Erreur création vente:', error);
        showError('Erreur lors de l\'enregistrement de la vente');
    }
}

// Calcul automatique du montant total
function calculateVenteMontant() {
    const quantite = parseFloat(document.getElementById('vente-quantite')?.value) || 0;
    const prixUnitaire = parseFloat(document.getElementById('vente-prix-unitaire')?.value) || 0;
    
    const montantTotal = quantite * prixUnitaire;
    
    // Mettre à jour l'affichage
    const montantElement = document.getElementById('vente-montant-total');
    if (montantElement) {
        montantElement.textContent = formatCurrency(montantTotal);
    }
}

// Utilitaires pour les ventes
function getVenteStatusClass(statut) {
    const classes = {
        'en_cours': 'badge-warning',
        'livre': 'badge-info',
        'paye': 'badge-success'
    };
    return classes[statut] || 'badge-secondary';
}

function getVenteStatusText(statut) {
    const texts = {
        'en_cours': 'En cours',
        'livre': 'Livré',
        'paye': 'Payé'
    };
    return texts[statut] || statut;
}

// Actions sur les ventes
function viewVente(id) {
    const vente = currentVentes.find(v => v.id === id);
    if (vente) {
        showVenteDetails(vente);
    }
}

function editVente(id) {
    const vente = currentVentes.find(v => v.id === id);
    if (vente) {
        populateVenteForm(vente);
        openModal('modal-vente');
    }
}

async function deleteVente(id) {
    if (!confirm('Êtes-vous sûr de vouloir supprimer cette vente ?')) {
        return;
    }
    
    try {
        if (isOnline) {
            const response = await fetch(`/api/ventes/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('authToken')}`
                }
            });
            
            if (response.ok) {
                showNotification('Vente supprimée avec succès', 'success');
            } else {
                throw new Error('Erreur lors de la suppression');
            }
        } else {
            await dbManager.delete('ventes', id);
            showNotification('Vente supprimée localement', 'info');
        }
        
        await loadVentesData();
        
    } catch (error) {
        console.error('Erreur suppression vente:', error);
        showError('Erreur lors de la suppression de la vente');
    }
}

// Chargement des statistiques des ventes
async function loadVentesStatistics() {
    try {
        const currentMonth = new Date().toISOString().substring(0, 7);
        const startOfMonth = `${currentMonth}-01`;
        const endOfMonth = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0)
            .toISOString().substring(0, 10);
        
        const ventesMonth = currentVentes.filter(v => 
            v.date_vente >= startOfMonth && v.date_vente <= endOfMonth
        );
        
        const stats = {
            total_ventes: ventesMonth.reduce((sum, v) => sum + v.montant_total, 0),
            total_marge: ventesMonth.reduce((sum, v) => sum + v.marge_brute, 0),
            nb_ventes: ventesMonth.length,
            quantite_totale: ventesMonth.reduce((sum, v) => sum + v.quantite, 0),
            prix_moyen: ventesMonth.length > 0 ? 
                ventesMonth.reduce((sum, v) => sum + v.prix_unitaire, 0) / ventesMonth.length : 0
        };
        
        displayVentesStatistics(stats);
        
    } catch (error) {
        console.error('Erreur calcul statistiques ventes:', error);
    }
}

function displayVentesStatistics(stats) {
    const statsContainer = document.querySelector('.ventes-statistics');
    if (statsContainer) {
        statsContainer.innerHTML = `
            <div class="stat-card">
                <div class="stat-value">${formatCurrency(stats.total_ventes)}</div>
                <div class="stat-label">Total ventes ce mois</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${formatCurrency(stats.total_marge)}</div>
                <div class="stat-label">Marge brute totale</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${stats.nb_ventes}</div>
                <div class="stat-label">Nombre de ventes</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">${formatCurrency(stats.prix_moyen)}</div>
                <div class="stat-label">Prix moyen</div>
            </div>
        `;
    }
}

function showVentesLoading(show) {
    const tableContainer = document.querySelector('#ventes-table-container');
    if (!tableContainer) return;
    
    if (show) {
        tableContainer.innerHTML = `
            <div class="loading-container">
                <div class="loading-spinner"></div>
                <p>Chargement des ventes...</p>
            </div>
        `;
    }
}

// Export des fonctions
window.loadVentesData = loadVentesData;
window.handleVenteSubmit = handleVenteSubmit;
window.calculateVenteMontant = calculateVenteMontant;
window.viewVente = viewVente;
window.editVente = editVente;
window.deleteVente = deleteVente;