<?php
// api/ventes.php - Endpoint pour la gestion des ventes

require_once 'config.php';

class VentesAPI {
    private $vente;
    private $user;
    
    public function __construct() {
        $this->vente = new Vente();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getVente($id);
                } else {
                    $this->getVentes();
                }
                break;
                
            case 'POST':
                $this->createVente();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateVente($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteVente($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getVentes() {
        $where = [];
        $orderBy = 'date_vente DESC';
        
        // Filtres optionnels
        if (isset($_GET['site_id'])) {
            $where['site_id'] = (int)$_GET['site_id'];
        }
        
        if (isset($_GET['client_id'])) {
            $where['client_id'] = (int)$_GET['client_id'];
        }
        
        $ventes = $this->vente->findAll($where, $orderBy, $_GET['limit'] ?? 100);
        
        // Enrichir avec les informations liées
        foreach ($ventes as &$vente) {
            $vente['site'] = $this->getSiteInfo($vente['site_id']);
            $vente['client'] = $this->getClientInfo($vente['client_id']);
            $vente['produit'] = $this->getProduitInfo($vente['produit_id']);
        }
        
        ApiResponse::success($ventes, 'Ventes récupérées');
    }
    
    private function getVente($id) {
        $vente = $this->vente->find($id);
        
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        // Enrichir avec les informations liées
        $vente['site'] = $this->getSiteInfo($vente['site_id']);
        $vente['client'] = $this->getClientInfo($vente['client_id']);
        $vente['produit'] = $this->getProduitInfo($vente['produit_id']);
        
        ApiResponse::success($vente, 'Vente récupérée');
    }
    
    private function createVente() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->vente->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        $data['utilisateur_id'] = $this->user['user_id'];
        
        // Calculer le montant total
        $venteObj = new Vente();
        foreach ($data as $key => $value) {
            $venteObj->$key = $value;
        }
        $venteObj->calculateTotals();
        $data['montant_total'] = $venteObj->montant_total;
        
        try {
            $this->vente->beginTransaction();
            
            // Vérifier le stock disponible
            $this->checkStockAvailable($data['site_id'], $data['produit_id'], $data['quantite']);
            
            // Créer la vente
            $id = $this->vente->create($data);
            
            // Mettre à jour le stock et calculer la marge
            $venteCreated = $this->vente->find($id);
            $venteObj = new Vente();
            foreach ($venteCreated as $key => $value) {
                $venteObj->$key = $value;
            }
            $venteObj->updateStock();
            
            // Créer la transaction financière
            $this->createTransaction($data['site_id'], 'entree', 'vente', $data['montant_total'], 
                "Vente - " . $this->getProduitName($data['produit_id']), $id);
            
            $this->vente->commit();
            
            ApiResponse::success(['id' => $id], 'Vente créée avec succès', 201);
            
        } catch (Exception $e) {
            $this->vente->rollback();
            Utils::logError('Erreur création vente: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création de la vente: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateVente($id) {
        $vente = $this->vente->find($id);
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->vente->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->vente->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Vente mise à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour vente: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour de la vente', 500);
        }
    }
    
    private function deleteVente($id) {
        $vente = $this->vente->find($id);
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        try {
            $success = $this->vente->delete($id);
            
            if ($success) {
                ApiResponse::success(null, 'Vente supprimée avec succès');
            } else {
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur suppression vente: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression de la vente', 500);
        }
    }
    
    // Méthodes utilitaires
    private function checkStockAvailable($siteId, $produitId, $quantite) {
        $stmt = $this->vente->db->prepare("SELECT quantite_actuelle FROM stocks WHERE site_id = ? AND produit_id = ?");
        $stmt->execute([$siteId, $produitId]);
        $stock = $stmt->fetch();
        
        if (!$stock || $stock['quantite_actuelle'] < $quantite) {
            throw new Exception('Stock insuffisant pour cette vente');
        }
    }
    
    private function createTransaction($siteId, $type, $categorie, $montant, $description, $referenceId) {
        $stmt = $this->vente->db->prepare("
            INSERT INTO transactions (site_id, type_transaction, categorie, montant, description, reference_id, date_transaction, utilisateur_id)
            VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?)
        ");
        
        return $stmt->execute([
            $siteId, $type, $categorie, $montant, $description, $referenceId, $this->user['user_id']
        ]);
    }
    
    private function getSiteInfo($siteId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, type_site FROM sites WHERE id = ?");
        $stmt->execute([$siteId]);
        return $stmt->fetch();
    }
    
    private function getClientInfo($clientId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, telephone, type_client FROM clients WHERE id = ?");
        $stmt->execute([$clientId]);
        return $stmt->fetch();
    }
    
    private function getProduitInfo($produitId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, unite_mesure, categorie FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        return $stmt->fetch();
    }
    
    private function getProduitName($produitId) {
        $stmt = $this->vente->db->prepare("SELECT nom FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        $result = $stmt->fetch();
        return $result ? $result['nom'] : 'Produit inconnu';
    }
}

try {
    $api = new VentesAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API ventes: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>