<?php
// api/sync.php - API de synchronisation pour PWA

require_once 'config.php';

class SyncAPI {
    private $db;
    private $user;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        
        switch ($method) {
            case 'POST':
                $this->handleSync();
                break;
            case 'GET':
                $this->handleDownload();
                break;
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function handleSync() {
        $input = Utils::parseJsonInput();
        
        if (!isset($input['action'])) {
            ApiResponse::error('Action requise', 400);
        }
        
        switch ($input['action']) {
            case 'sync_item':
                $this->syncItem($input);
                break;
            case 'sync_batch':
                $this->syncBatch($input);
                break;
            case 'get_pending':
                $this->getPendingSync();
                break;
            default:
                ApiResponse::error('Action non reconnue', 400);
        }
    }
    
    private function syncItem($data) {
        if (!isset($data['table_name'], $data['record_id'], $data['action'], $data['data'])) {
            ApiResponse::error('Données de synchronisation incomplètes', 400);
        }
        
        $tableName = $data['table_name'];
        $recordId = $data['record_id'];
        $action = $data['action'];
        $recordData = json_decode($data['data'], true);
        
        try {
            $this->db->beginTransaction();
            
            $result = $this->processSyncItem($tableName, $recordId, $action, $recordData);
            
            if ($result['success']) {
                // Marquer comme synchronisé dans la table sync_status si elle existe
                $this->markAsSynced($data['id'] ?? null);
                $this->db->commit();
                ApiResponse::success($result['data'], 'Synchronisation réussie');
            } else {
                $this->db->rollback();
                ApiResponse::error($result['message'], 400);
            }
            
        } catch (Exception $e) {
            $this->db->rollback();
            Utils::logError('Erreur sync item: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la synchronisation: ' . $e->getMessage(), 500);
        }
    }
    
    private function processSyncItem($tableName, $recordId, $action, $recordData) {
        // Validation de la table
        $allowedTables = [
            'sites', 'budgets', 'achats', 'ventes', 'stocks', 
            'fournisseurs', 'clients', 'produits', 'transactions'
        ];
        
        if (!in_array($tableName, $allowedTables)) {
            return ['success' => false, 'message' => 'Table non autorisée'];
        }
        
        // Récupérer le modèle approprié
        $modelClass = ucfirst($tableName);
        if ($tableName === 'fournisseurs') $modelClass = 'Fournisseur';
        if ($tableName === 'clients') $modelClass = 'Client';
        if ($tableName === 'produits') $modelClass = 'Produit';
        
        try {
            $model = new $modelClass();
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Modèle non trouvé: ' . $modelClass];
        }
        
        switch ($action) {
            case 'insert':
                return $this->handleInsert($model, $recordData);
                
            case 'update':
                return $this->handleUpdate($model, $recordId, $recordData);
                
            case 'delete':
                return $this->handleDelete($model, $recordId);
                
            default:
                return ['success' => false, 'message' => 'Action non supportée'];
        }
    }
    
    private function handleInsert($model, $data) {
        // Validation des données
        $errors = $model->validate($data);
        if (!empty($errors)) {
            return ['success' => false, 'message' => 'Erreurs de validation', 'errors' => $errors];
        }
        
        // Nettoyer les données
        $cleanData = Utils::sanitizeInput($data);
        
        // Créer l'enregistrement
        $id = $model->create($cleanData);
        
        // Post-traitement spécifique
        $this->postProcessCreate($model, $id, $cleanData);
        
        return ['success' => true, 'data' => ['id' => $id], 'message' => 'Enregistrement créé'];
    }
    
    private function handleUpdate($model, $id, $data) {
        // Vérifier que l'enregistrement existe
        $existing = $model->find($id);
        if (!$existing) {
            return ['success' => false, 'message' => 'Enregistrement non trouvé'];
        }
        
        // Validation des données
        $errors = $model->validate($data, true);
        if (!empty($errors)) {
            return ['success' => false, 'message' => 'Erreurs de validation', 'errors' => $errors];
        }
        
        // Nettoyer les données
        $cleanData = Utils::sanitizeInput($data);
        
        // Mettre à jour
        $success = $model->update($id, $cleanData);
        
        if ($success) {
            // Post-traitement spécifique
            $this->postProcessUpdate($model, $id, $cleanData, $existing);
            return ['success' => true, 'message' => 'Enregistrement mis à jour'];
        } else {
            return ['success' => false, 'message' => 'Échec de la mise à jour'];
        }
    }
    
    private function handleDelete($model, $id) {
        // Vérifier que l'enregistrement existe
        $existing = $model->find($id);
        if (!$existing) {
            return ['success' => false, 'message' => 'Enregistrement non trouvé'];
        }
        
        // Vérifications de contraintes métier
        $canDelete = $this->canDelete($model, $id);
        if (!$canDelete['allowed']) {
            return ['success' => false, 'message' => $canDelete['reason']];
        }
        
        // Supprimer
        $success = $model->delete($id);
        
        if ($success) {
            return ['success' => true, 'message' => 'Enregistrement supprimé'];
        } else {
            return ['success' => false, 'message' => 'Échec de la suppression'];
        }
    }
    
    private function postProcessCreate($model, $id, $data) {
        $tableName = $model->table ?? '';
        
        switch ($tableName) {
            case 'achats':
                // Mettre à jour le stock
                $achat = $model->find($id);
                if ($achat) {
                    $achatObj = new Achat();
                    foreach ($achat as $key => $value) {
                        $achatObj->$key = $value;
                    }
                    $achatObj->updateStock();
                    
                    // Créer la transaction financière
                    $this->createTransaction($achat['site_id'], 'sortie', 'achat', $achat['cout_total'], 
                        "Achat - " . $this->getProduitName($achat['produit_id']), $id);
                }
                break;
                
            case 'ventes':
                // Mettre à jour le stock
                $vente = $model->find($id);
                if ($vente) {
                    $venteObj = new Vente();
                    foreach ($vente as $key => $value) {
                        $venteObj->$key = $value;
                    }
                    $venteObj->updateStock();
                    
                    // Créer la transaction financière
                    $this->createTransaction($vente['site_id'], 'entree', 'vente', $vente['montant_total'],
                        "Vente - " . $this->getProduitName($vente['produit_id']), $id);
                }
                break;
        }
    }
    
    private function postProcessUpdate($model, $id, $newData, $oldData) {
        // Gérer les mises à jour qui impactent les stocks ou transactions
        $tableName = $model->table ?? '';
        
        switch ($tableName) {
            case 'achats':
                // Recalculer le stock si les quantités ont changé
                if (isset($newData['quantite']) && $newData['quantite'] != $oldData['quantite']) {
                    // Logique de mise à jour complexe à implémenter
                }
                break;
                
            case 'ventes':
                // Recalculer le stock si les quantités ont changé
                if (isset($newData['quantite']) && $newData['quantite'] != $oldData['quantite']) {
                    // Logique de mise à jour complexe à implémenter
                }
                break;
        }
    }
    
    private function canDelete($model, $id) {
        $tableName = $model->table ?? '';
        
        switch ($tableName) {
            case 'sites':
                // Vérifier s'il y a des achats/ventes associés
                $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM achats WHERE site_id = ?");
                $stmt->execute([$id]);
                if ($stmt->fetch()['count'] > 0) {
                    return ['allowed' => false, 'reason' => 'Impossible de supprimer un site avec des achats associés'];
                }
                
                $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM ventes WHERE site_id = ?");
                $stmt->execute([$id]);
                if ($stmt->fetch()['count'] > 0) {
                    return ['allowed' => false, 'reason' => 'Impossible de supprimer un site avec des ventes associées'];
                }
                break;
                
            case 'produits':
                // Vérifier s'il y a du stock
                $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM stocks WHERE produit_id = ? AND quantite_actuelle > 0");
                $stmt->execute([$id]);
                if ($stmt->fetch()['count'] > 0) {
                    return ['allowed' => false, 'reason' => 'Impossible de supprimer un produit avec du stock'];
                }
                break;
        }
        
        return ['allowed' => true];
    }
    
    private function createTransaction($siteId, $type, $categorie, $montant, $description, $referenceId = null) {
        $stmt = $this->db->prepare("
            INSERT INTO transactions (site_id, type_transaction, categorie, montant, description, reference_id, date_transaction, utilisateur_id)
            VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?)
        ");
        
        return $stmt->execute([
            $siteId, $type, $categorie, $montant, $description, $referenceId, $this->user['user_id']
        ]);
    }
    
    private function getProduitName($produitId) {
        $stmt = $this->db->prepare("SELECT nom FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        $result = $stmt->fetch();
        return $result ? $result['nom'] : 'Produit inconnu';
    }
    
    private function markAsSynced($syncId) {
        if (!$syncId) return;
        
        $stmt = $this->db->prepare("UPDATE sync_status SET synced = 1, date_sync = NOW() WHERE id = ?");
        $stmt->execute([$syncId]);
    }
    
    private function syncBatch($data) {
        if (!isset($data['items']) || !is_array($data['items'])) {
            ApiResponse::error('Items requis pour la synchronisation par lot', 400);
        }
        
        $results = [];
        $successCount = 0;
        $errorCount = 0;
        
        try {
            $this->db->beginTransaction();
            
            foreach ($data['items'] as $item) {
                try {
                    $result = $this->processSyncItem(
                        $item['table_name'],
                        $item['record_id'],
                        $item['action'],
                        json_decode($item['data'], true)
                    );
                    
                    if ($result['success']) {
                        $successCount++;
                        $this->markAsSynced($item['id'] ?? null);
                    } else {
                        $errorCount++;
                    }
                    
                    $results[] = [
                        'item_id' => $item['id'] ?? null,
                        'success' => $result['success'],
                        'message' => $result['message'] ?? ''
                    ];
                    
                } catch (Exception $e) {
                    $errorCount++;
                    $results[] = [
                        'item_id' => $item['id'] ?? null,
                        'success' => false,
                        'message' => $e->getMessage()
                    ];
                }
            }
            
            if ($errorCount === 0) {
                $this->db->commit();
            } else {
                $this->db->rollback();
            }
            
            ApiResponse::success([
                'total_items' => count($data['items']),
                'success_count' => $successCount,
                'error_count' => $errorCount,
                'results' => $results
            ], 'Synchronisation par lot terminée');
            
        } catch (Exception $e) {
            $this->db->rollback();
            Utils::logError('Erreur sync batch: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la synchronisation par lot', 500);
        }
    }
    
    private function getPendingSync() {
        // Retourner les éléments en attente de synchronisation depuis le serveur
        $stmt = $this->db->prepare("
            SELECT table_name, record_id, action, data, date_creation
            FROM sync_status 
            WHERE synced = 0 
            ORDER BY date_creation ASC 
            LIMIT 100
        ");
        $stmt->execute();
        $pending = $stmt->fetchAll();
        
        ApiResponse::success($pending, 'Éléments en attente récupérés');
    }
    
    private function handleDownload() {
        $lastSync = $_GET['last_sync'] ?? null;
        $tables = $_GET['tables'] ?? 'all';
        
        $data = [];
        
        if ($tables === 'all' || strpos($tables, 'sites') !== false) {
            $data['sites'] = $this->getTableData('sites', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'budgets') !== false) {
            $data['budgets'] = $this->getTableData('budgets', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'produits') !== false) {
            $data['produits'] = $this->getTableData('produits', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'fournisseurs') !== false) {
            $data['fournisseurs'] = $this->getTableData('fournisseurs', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'clients') !== false) {
            $data['clients'] = $this->getTableData('clients', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'achats') !== false) {
            $data['achats'] = $this->getTableData('achats', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'ventes') !== false) {
            $data['ventes'] = $this->getTableData('ventes', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'stocks') !== false) {
            $data['stocks'] = $this->getTableData('stocks', $lastSync);
        }
        
        if ($tables === 'all' || strpos($tables, 'transactions') !== false) {
            $data['transactions'] = $this->getTableData('transactions', $lastSync);
        }
        
        ApiResponse::success([
            'data' => $data,
            'sync_timestamp' => date('c'),
            'total_records' => array_sum(array_map('count', $data))
        ], 'Données téléchargées');
    }
    
    private function getTableData($tableName, $lastSync = null) {
        $sql = "SELECT * FROM $tableName";
        $params = [];
        
        if ($lastSync) {
            $sql .= " WHERE last_modified > ?";
            $params[] = $lastSync;
        }
        
        $sql .= " ORDER BY id ASC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }
}

// Point d'entrée de l'API
try {
    $api = new SyncAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API sync: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>