<?php
// api/sites.php - Endpoint pour la gestion des sites

require_once 'config.php';

class SitesAPI {
    private $site;
    private $user;
    
    public function __construct() {
        $this->site = new Site();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getSite($id);
                } else {
                    $this->getSites();
                }
                break;
                
            case 'POST':
                $this->createSite();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateSite($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteSite($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getSites() {
        $sites = $this->site->findAll();
        
        // Enrichir avec les données de budget
        foreach ($sites as &$site) {
            $siteObj = new Site();
            $siteObj->id = $site['id'];
            $site['budgets'] = $siteObj->getBudgets();
            $site['budget_total'] = $siteObj->getTotalBudget();
            $site['stats_mois'] = $siteObj->getMonthlyStats(date('Y'), date('m'));
        }
        
        ApiResponse::success($sites, 'Sites récupérés');
    }
    
    private function getSite($id) {
        $site = $this->site->find($id);
        
        if (!$site) {
            ApiResponse::error('Site non trouvé', 404);
        }
        
        // Enrichir avec les données détaillées
        $siteObj = new Site();
        $siteObj->id = $id;
        $site['budgets'] = $siteObj->getBudgets();
        $site['budget_total'] = $siteObj->getTotalBudget();
        $site['stats_mois'] = $siteObj->getMonthlyStats(date('Y'), date('m'));
        $site['stocks'] = $siteObj->getStocks();
        
        ApiResponse::success($site, 'Site récupéré');
    }
    
    private function createSite() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->site->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        $data['responsable_id'] = $this->user['user_id'];
        
        try {
            $id = $this->site->create($data);
            
            // Créer un budget initial si spécifié
            if (isset($data['budget_initial']) && $data['budget_initial'] > 0) {
                $budget = new Budget();
                $budgetData = [
                    'site_id' => $id,
                    'montant_initial' => $data['budget_initial'],
                    'montant_actuel' => $data['budget_initial'],
                    'periode_debut' => date('Y-m-d'),
                    'periode_fin' => date('Y-m-d', strtotime('+1 year')),
                    'statut' => 'actif'
                ];
                $budget->create($budgetData);
            }
            
            ApiResponse::success(['id' => $id], 'Site créé avec succès', 201);
            
        } catch (Exception $e) {
            Utils::logError('Erreur création site: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création du site', 500);
        }
    }
    
    private function updateSite($id) {
        $site = $this->site->find($id);
        if (!$site) {
            ApiResponse::error('Site non trouvé', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->site->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->site->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Site mis à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour site: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour du site', 500);
        }
    }
    
    private function deleteSite($id) {
        $site = $this->site->find($id);
        if (!$site) {
            ApiResponse::error('Site non trouvé', 404);
        }
        
        // Vérifier les contraintes
        // (cette logique est aussi dans le SyncAPI)
        
        try {
            $success = $this->site->delete($id);
            
            if ($success) {
                ApiResponse::success(null, 'Site supprimé avec succès');
            } else {
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur suppression site: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression du site', 500);
        }
    }
}

try {
    $api = new SitesAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API sites: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>

<?php
// api/achats.php - Endpoint pour la gestion des achats

require_once 'config.php';

class AchatsAPI {
    private $achat;
    private $user;
    
    public function __construct() {
        $this->achat = new Achat();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getAchat($id);
                } else {
                    $this->getAchats();
                }
                break;
                
            case 'POST':
                $this->createAchat();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateAchat($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteAchat($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getAchats() {
        $where = [];
        $orderBy = 'date_achat DESC';
        
        // Filtres optionnels
        if (isset($_GET['site_id'])) {
            $where['site_id'] = (int)$_GET['site_id'];
        }
        
        if (isset($_GET['fournisseur_id'])) {
            $where['fournisseur_id'] = (int)$_GET['fournisseur_id'];
        }
        
        if (isset($_GET['date_debut']) && isset($_GET['date_fin'])) {
            // Pour les plages de dates, utiliser une requête personnalisée
            $achats = $this->getAchatsByDateRange($_GET['date_debut'], $_GET['date_fin'], $where);
        } else {
            $achats = $this->achat->findAll($where, $orderBy, $_GET['limit'] ?? 100);
        }
        
        // Enrichir avec les informations liées
        foreach ($achats as &$achat) {
            $achat['site'] = $this->getSiteInfo($achat['site_id']);
            $achat['fournisseur'] = $this->getFournisseurInfo($achat['fournisseur_id']);
            $achat['produit'] = $this->getProduitInfo($achat['produit_id']);
        }
        
        ApiResponse::success($achats, 'Achats récupérés');
    }
    
    private function getAchatsByDateRange($dateDebut, $dateFin, $additionalWhere = []) {
        $sql = "SELECT * FROM achats WHERE date_achat BETWEEN ? AND ?";
        $params = [$dateDebut, $dateFin];
        
        foreach ($additionalWhere as $field => $value) {
            $sql .= " AND $field = ?";
            $params[] = $value;
        }
        
        $sql .= " ORDER BY date_achat DESC";
        
        $stmt = $this->achat->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }
    
    private function getAchat($id) {
        $achat = $this->achat->find($id);
        
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        // Enrichir avec les informations liées
        $achat['site'] = $this->getSiteInfo($achat['site_id']);
        $achat['fournisseur'] = $this->getFournisseurInfo($achat['fournisseur_id']);
        $achat['produit'] = $this->getProduitInfo($achat['produit_id']);
        
        ApiResponse::success($achat, 'Achat récupéré');
    }
    
    private function createAchat() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->achat->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        $data['utilisateur_id'] = $this->user['user_id'];
        
        // Calculer les totaux
        $achatObj = new Achat();
        foreach ($data as $key => $value) {
            $achatObj->$key = $value;
        }
        $achatObj->calculateTotals();
        
        $data['montant_total'] = $achatObj->montant_total;
        $data['cout_total'] = $achatObj->cout_total;
        
        try {
            $this->achat->beginTransaction();
            
            // Vérifier le budget disponible
            $this->checkBudgetAvailable($data['site_id'], $data['cout_total']);
            
            // Créer l'achat
            $id = $this->achat->create($data);
            
            // Mettre à jour le stock
            $achatCreated = $this->achat->find($id);
            $achatObj = new Achat();
            foreach ($achatCreated as $key => $value) {
                $achatObj->$key = $value;
            }
            $achatObj->updateStock();
            
            // Mettre à jour le budget
            $this->updateBudget($data['site_id'], $data['cout_total'], 'subtract');
            
            // Créer la transaction financière
            $this->createTransaction($data['site_id'], 'sortie', 'achat', $data['cout_total'], 
                "Achat - " . $this->getProduitName($data['produit_id']), $id);
            
            $this->achat->commit();
            
            ApiResponse::success(['id' => $id], 'Achat créé avec succès', 201);
            
        } catch (Exception $e) {
            $this->achat->rollback();
            Utils::logError('Erreur création achat: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création de l\'achat: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateAchat($id) {
        $achat = $this->achat->find($id);
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->achat->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->achat->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Achat mis à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour achat: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour de l\'achat', 500);
        }
    }
    
    private function deleteAchat($id) {
        $achat = $this->achat->find($id);
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        try {
            $this->achat->beginTransaction();
            
            // Logique de suppression complexe (inverser les mouvements de stock, etc.)
            // Pour l'instant, simple suppression
            $success = $this->achat->delete($id);
            
            if ($success) {
                $this->achat->commit();
                ApiResponse::success(null, 'Achat supprimé avec succès');
            } else {
                $this->achat->rollback();
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            $this->achat->rollback();
            Utils::logError('Erreur suppression achat: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression de l\'achat', 500);
        }
    }
    
    // Méthodes utilitaires
    private function checkBudgetAvailable($siteId, $montant) {
        $budget = new Budget();
        $budgets = $budget->findAll(['site_id' => $siteId, 'statut' => 'actif']);
        
        $budgetTotal = array_sum(array_column($budgets, 'montant_actuel'));
        
        if ($budgetTotal < $montant) {
            throw new Exception('Budget insuffisant pour cet achat');
        }
    }
    
    private function updateBudget($siteId, $montant, $operation) {
        $budget = new Budget();
        $budgets = $budget->findAll(['site_id' => $siteId, 'statut' => 'actif'], 'date_creation ASC');
        
        $montantRestant = $montant;
        
        foreach ($budgets as $budgetData) {
            if ($montantRestant <= 0) break;
            
            $budgetObj = new Budget();
            foreach ($budgetData as $key => $value) {
                $budgetObj->$key = $value;
            }
            
            if ($operation === 'subtract') {
                $aDeduire = min($montantRestant, $budgetData['montant_actuel']);
                $budgetObj->updateAmount($aDeduire, 'subtract');
                $montantRestant -= $aDeduire;
            }
        }
    }
    
    private function createTransaction($siteId, $type, $categorie, $montant, $description, $referenceId) {
        $stmt = $this->achat->db->prepare("
            INSERT INTO transactions (site_id, type_transaction, categorie, montant, description, reference_id, date_transaction, utilisateur_id)
            VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?)
        ");
        
        return $stmt->execute([
            $siteId, $type, $categorie, $montant, $description, $referenceId, $this->user['user_id']
        ]);
    }
    
    private function getSiteInfo($siteId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, type_site FROM sites WHERE id = ?");
        $stmt->execute([$siteId]);
        return $stmt->fetch();
    }
    
    private function getFournisseurInfo($fournisseurId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, telephone FROM fournisseurs WHERE id = ?");
        $stmt->execute([$fournisseurId]);
        return $stmt->fetch();
    }
    
    private function getProduitInfo($produitId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, unite_mesure, categorie FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        return $stmt->fetch();
    }
    
    private function getProduitName($produitId) {
        $stmt = $this->achat->db->prepare("SELECT nom FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        $result = $stmt->fetch();
        return $result ? $result['nom'] : 'Produit inconnu';
    }
}

try {
    $api = new AchatsAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API achats: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>

<?php
// api/ventes.php - Endpoint pour la gestion des ventes

require_once 'config.php';

class VentesAPI {
    private $vente;
    private $user;
    
    public function __construct() {
        $this->vente = new Vente();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getVente($id);
                } else {
                    $this->getVentes();
                }
                break;
                
            case 'POST':
                $this->createVente();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateVente($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteVente($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getVentes() {
        $where = [];
        $orderBy = 'date_vente DESC';
        
        // Filtres optionnels
        if (isset($_GET['site_id'])) {
            $where['site_id'] = (int)$_GET['site_id'];
        }
        
        if (isset($_GET['client_id'])) {
            $where['client_id'] = (int)$_GET['client_id'];
        }
        
        $ventes = $this->vente->findAll($where, $orderBy, $_GET['limit'] ?? 100);
        
        // Enrichir avec les informations liées
        foreach ($ventes as &$vente) {
            $vente['site'] = $this->getSiteInfo($vente['site_id']);
            $vente['client'] = $this->getClientInfo($vente['client_id']);
            $vente['produit'] = $this->getProduitInfo($vente['produit_id']);
        }
        
        ApiResponse::success($ventes, 'Ventes récupérées');
    }
    
    private function getVente($id) {
        $vente = $this->vente->find($id);
        
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        // Enrichir avec les informations liées
        $vente['site'] = $this->getSiteInfo($vente['site_id']);
        $vente['client'] = $this->getClientInfo($vente['client_id']);
        $vente['produit'] = $this->getProduitInfo($vente['produit_id']);
        
        ApiResponse::success($vente, 'Vente récupérée');
    }
    
    private function createVente() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->vente->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        $data['utilisateur_id'] = $this->user['user_id'];
        
        // Calculer le montant total
        $venteObj = new Vente();
        foreach ($data as $key => $value) {
            $venteObj->$key = $value;
        }
        $venteObj->calculateTotals();
        $data['montant_total'] = $venteObj->montant_total;
        
        try {
            $this->vente->beginTransaction();
            
            // Vérifier le stock disponible
            $this->checkStockAvailable($data['site_id'], $data['produit_id'], $data['quantite']);
            
            // Créer la vente
            $id = $this->vente->create($data);
            
            // Mettre à jour le stock et calculer la marge
            $venteCreated = $this->vente->find($id);
            $venteObj = new Vente();
            foreach ($venteCreated as $key => $value) {
                $venteObj->$key = $value;
            }
            $venteObj->updateStock();
            
            // Créer la transaction financière
            $this->createTransaction($data['site_id'], 'entree', 'vente', $data['montant_total'], 
                "Vente - " . $this->getProduitName($data['produit_id']), $id);
            
            $this->vente->commit();
            
            ApiResponse::success(['id' => $id], 'Vente créée avec succès', 201);
            
        } catch (Exception $e) {
            $this->vente->rollback();
            Utils::logError('Erreur création vente: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création de la vente: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateVente($id) {
        $vente = $this->vente->find($id);
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->vente->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->vente->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Vente mise à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour vente: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour de la vente', 500);
        }
    }
    
    private function deleteVente($id) {
        $vente = $this->vente->find($id);
        if (!$vente) {
            ApiResponse::error('Vente non trouvée', 404);
        }
        
        try {
            $success = $this->vente->delete($id);
            
            if ($success) {
                ApiResponse::success(null, 'Vente supprimée avec succès');
            } else {
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur suppression vente: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression de la vente', 500);
        }
    }
    
    // Méthodes utilitaires
    private function checkStockAvailable($siteId, $produitId, $quantite) {
        $stmt = $this->vente->db->prepare("SELECT quantite_actuelle FROM stocks WHERE site_id = ? AND produit_id = ?");
        $stmt->execute([$siteId, $produitId]);
        $stock = $stmt->fetch();
        
        if (!$stock || $stock['quantite_actuelle'] < $quantite) {
            throw new Exception('Stock insuffisant pour cette vente');
        }
    }
    
    private function createTransaction($siteId, $type, $categorie, $montant, $description, $referenceId) {
        $stmt = $this->vente->db->prepare("
            INSERT INTO transactions (site_id, type_transaction, categorie, montant, description, reference_id, date_transaction, utilisateur_id)
            VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?)
        ");
        
        return $stmt->execute([
            $siteId, $type, $categorie, $montant, $description, $referenceId, $this->user['user_id']
        ]);
    }
    
    private function getSiteInfo($siteId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, type_site FROM sites WHERE id = ?");
        $stmt->execute([$siteId]);
        return $stmt->fetch();
    }
    
    private function getClientInfo($clientId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, telephone, type_client FROM clients WHERE id = ?");
        $stmt->execute([$clientId]);
        return $stmt->fetch();
    }
    
    private function getProduitInfo($produitId) {
        $stmt = $this->vente->db->prepare("SELECT id, nom, unite_mesure, categorie FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        return $stmt->fetch();
    }
    
    private function getProduitName($produitId) {
        $stmt = $this->vente->db->prepare("SELECT nom FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        $result = $stmt->fetch();
        return $result ? $result['nom'] : 'Produit inconnu';
    }
}

try {
    $api = new VentesAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API ventes: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>