<?php
// api/reporting.php - Endpoint pour les rapports et analyses

require_once 'config.php';

class ReportingAPI {
    private $db;
    private $user;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $reportType = $pathParts[2] ?? null;
        
        if ($method !== 'GET') {
            ApiResponse::error('Seule la méthode GET est supportée pour les rapports', 405);
        }
        
        switch ($reportType) {
            case 'dashboard':
                $this->getDashboardData();
                break;
            case 'rentabilite':
                $this->getRentabiliteReport();
                break;
            case 'stocks':
                $this->getStocksReport();
                break;
            case 'transactions':
                $this->getTransactionsReport();
                break;
            case 'performance':
                $this->getPerformanceReport();
                break;
            case 'evolution':
                $this->getEvolutionReport();
                break;
            default:
                ApiResponse::error('Type de rapport non reconnu', 404);
        }
    }
    
    private function getDashboardData() {
        try {
            $data = [];
            
            // KPI globaux
            $data['kpi'] = $this->getGlobalKPI();
            
            // Évolution sur 12 mois
            $data['evolution_mensuelle'] = $this->getEvolutionMensuelle();
            
            // Top sites
            $data['top_sites'] = $this->getTopSites();
            
            // Top produits
            $data['top_produits'] = $this->getTopProduits();
            
            // Transactions récentes
            $data['transactions_recentes'] = $this->getTransactionsRecentes();
            
            ApiResponse::success($data, 'Données du tableau de bord récupérées');
            
        } catch (Exception $e) {
            Utils::logError('Erreur dashboard: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du tableau de bord', 500);
        }
    }
    
    private function getGlobalKPI() {
        $currentMonth = date('Y-m');
        $startOfMonth = $currentMonth . '-01';
        $endOfMonth = date('Y-m-t');
        
        // Budget total actuel
        $stmt = $this->db->prepare("SELECT SUM(montant_actuel) as total FROM budgets WHERE statut = 'actif'");
        $stmt->execute();
        $budgetTotal = $stmt->fetch()['total'] ?? 0;
        
        // Achats du mois
        $stmt = $this->db->prepare("SELECT SUM(cout_total) as total FROM achats WHERE date_achat BETWEEN ? AND ?");
        $stmt->execute([$startOfMonth, $endOfMonth]);
        $achatsMonth = $stmt->fetch()['total'] ?? 0;
        
        // Ventes du mois
        $stmt = $this->db->prepare("SELECT SUM(montant_total) as total, SUM(marge_brute) as marge FROM ventes WHERE date_vente BETWEEN ? AND ?");
        $stmt->execute([$startOfMonth, $endOfMonth]);
        $ventesData = $stmt->fetch();
        $ventesMonth = $ventesData['total'] ?? 0;
        $margeMonth = $ventesData['marge'] ?? 0;
        
        // ROI
        $roi = $achatsMonth > 0 ? ($margeMonth / $achatsMonth) * 100 : 0;
        
        return [
            'budget_total' => $budgetTotal,
            'achats_mois' => $achatsMonth,
            'ventes_mois' => $ventesMonth,
            'marge_mois' => $margeMonth,
            'roi_mois' => $roi
        ];
    }
    
    private function getEvolutionMensuelle() {
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = new DateTime();
            $date->modify("-$i months");
            $months[] = $date->format('Y-m');
        }
        
        $evolution = [];
        
        foreach ($months as $month) {
            $startDate = $month . '-01';
            $endDate = date('Y-m-t', strtotime($startDate));
            
            // Achats
            $stmt = $this->db->prepare("SELECT SUM(cout_total) as total FROM achats WHERE date_achat BETWEEN ? AND ?");
            $stmt->execute([$startDate, $endDate]);
            $achats = $stmt->fetch()['total'] ?? 0;
            
            // Ventes
            $stmt = $this->db->prepare("SELECT SUM(montant_total) as total, SUM(marge_brute) as marge FROM ventes WHERE date_vente BETWEEN ? AND ?");
            $stmt->execute([$startDate, $endDate]);
            $ventesData = $stmt->fetch();
            
            $evolution[] = [
                'mois' => $month,
                'achats' => $achats,
                'ventes' => $ventesData['total'] ?? 0,
                'marge' => $ventesData['marge'] ?? 0
            ];
        }
        
        return $evolution;
    }
    
    private function getTopSites() {
        $currentMonth = date('Y-m');
        $startOfMonth = $currentMonth . '-01';
        $endOfMonth = date('Y-m-t');
        
        $stmt = $this->db->prepare("
            SELECT s.id, s.nom, 
                   SUM(v.montant_total) as ventes_mois,
                   SUM(v.marge_brute) as marge_mois,
                   COUNT(v.id) as nb_ventes
            FROM sites s
            LEFT JOIN ventes v ON s.id = v.site_id AND v.date_vente BETWEEN ? AND ?
            GROUP BY s.id, s.nom
            ORDER BY ventes_mois DESC
            LIMIT 5
        ");
        $stmt->execute([$startOfMonth, $endOfMonth]);
        
        return $stmt->fetchAll();
    }
    
    private function getTopProduits() {
        $currentMonth = date('Y-m');
        $startOfMonth = $currentMonth . '-01';
        $endOfMonth = date('Y-m-t');
        
        $stmt = $this->db->prepare("
            SELECT p.id, p.nom, p.unite_mesure,
                   SUM(v.quantite) as quantite_vendue,
                   SUM(v.montant_total) as chiffre_affaires,
                   SUM(v.marge_brute) as marge_totale
            FROM produits p
            JOIN ventes v ON p.id = v.produit_id
            WHERE v.date_vente BETWEEN ? AND ?
            GROUP BY p.id, p.nom, p.unite_mesure
            ORDER BY chiffre_affaires DESC
            LIMIT 10
        ");
        $stmt->execute([$startOfMonth, $endOfMonth]);
        
        return $stmt->fetchAll();
    }
    
    private function getTransactionsRecentes() {
        $stmt = $this->db->prepare("
            SELECT t.*, s.nom as site_nom
            FROM transactions t
            JOIN sites s ON t.site_id = s.id
            ORDER BY t.date_creation DESC
            LIMIT 10
        ");
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    private function getRentabiliteReport() {
        $dateDebut = $_GET['date_debut'] ?? date('Y-m-01');
        $dateFin = $_GET['date_fin'] ?? date('Y-m-t');
        $siteId = $_GET['site_id'] ?? null;
        
        try {
            $where = "v.date_vente BETWEEN ? AND ?";
            $params = [$dateDebut, $dateFin];
            
            if ($siteId) {
                $where .= " AND v.site_id = ?";
                $params[] = $siteId;
            }
            
            // Rentabilité par site
            $stmt = $this->db->prepare("
                SELECT s.id, s.nom as site_nom,
                       SUM(v.montant_total) as total_ventes,
                       SUM(v.cout_achat) as total_couts,
                       SUM(v.marge_brute) as marge_brute,
                       COUNT(v.id) as nb_ventes,
                       CASE 
                           WHEN SUM(v.cout_achat) > 0 
                           THEN (SUM(v.marge_brute) / SUM(v.cout_achat)) * 100 
                           ELSE 0 
                       END as roi_pourcentage
                FROM sites s
                LEFT JOIN ventes v ON s.id = v.site_id AND $where
                GROUP BY s.id, s.nom
                ORDER BY marge_brute DESC
            ");
            $stmt->execute($params);
            $rentabiliteParSite = $stmt->fetchAll();
            
            // Rentabilité par produit
            $stmt = $this->db->prepare("
                SELECT p.id, p.nom as produit_nom,
                       SUM(v.quantite) as quantite_vendue,
                       SUM(v.montant_total) as total_ventes,
                       SUM(v.cout_achat) as total_couts,
                       SUM(v.marge_brute) as marge_brute,
                       AVG(v.prix_unitaire) as prix_moyen,
                       CASE 
                           WHEN SUM(v.cout_achat) > 0 
                           THEN (SUM(v.marge_brute) / SUM(v.cout_achat)) * 100 
                           ELSE 0 
                       END as roi_pourcentage
                FROM produits p
                JOIN ventes v ON p.id = v.produit_id
                WHERE $where
                GROUP BY p.id, p.nom
                ORDER BY marge_brute DESC
            ");
            $stmt->execute($params);
            $rentabiliteParProduit = $stmt->fetchAll();
            
            ApiResponse::success([
                'periode' => ['debut' => $dateDebut, 'fin' => $dateFin],
                'par_site' => $rentabiliteParSite,
                'par_produit' => $rentabiliteParProduit
            ], 'Rapport de rentabilité généré');
            
        } catch (Exception $e) {
            Utils::logError('Erreur rapport rentabilité: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du rapport de rentabilité', 500);
        }
    }
    
    private function getStocksReport() {
        try {
            // Stocks actuels par site
            $stmt = $this->db->prepare("
                SELECT s.nom as site_nom, p.nom as produit_nom, p.unite_mesure,
                       st.quantite_actuelle, st.valeur_stock, st.prix_moyen_pondere
                FROM stocks st
                JOIN sites s ON st.site_id = s.id
                JOIN produits p ON st.produit_id = p.id
                WHERE st.quantite_actuelle > 0
                ORDER BY s.nom, p.nom
            ");
            $stmt->execute();
            $stocksDetailles = $stmt->fetchAll();
            
            // Valeur totale des stocks par site
            $stmt = $this->db->prepare("
                SELECT s.id, s.nom as site_nom,
                       SUM(st.valeur_stock) as valeur_totale,
                       COUNT(st.id) as nb_produits
                FROM sites s
                LEFT JOIN stocks st ON s.id = st.site_id AND st.quantite_actuelle > 0
                GROUP BY s.id, s.nom
                ORDER BY valeur_totale DESC
            ");
            $stmt->execute();
            $valeursParSite = $stmt->fetchAll();
            
            // Mouvements récents
            $stmt = $this->db->prepare("
                SELECT ms.*, s.nom as site_nom, p.nom as produit_nom
                FROM mouvements_stock ms
                JOIN stocks st ON ms.stock_id = st.id
                JOIN sites s ON st.site_id = s.id
                JOIN produits p ON st.produit_id = p.id
                ORDER BY ms.date_mouvement DESC
                LIMIT 50
            ");
            $stmt->execute();
            $mouvementsRecents = $stmt->fetchAll();
            
            ApiResponse::success([
                'stocks_detailles' => $stocksDetailles,
                'valeurs_par_site' => $valeursParSite,
                'mouvements_recents' => $mouvementsRecents
            ], 'Rapport des stocks généré');
            
        } catch (Exception $e) {
            Utils::logError('Erreur rapport stocks: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du rapport des stocks', 500);
        }
    }
    
    private function getTransactionsReport() {
        $dateDebut = $_GET['date_debut'] ?? date('Y-m-01');
        $dateFin = $_GET['date_fin'] ?? date('Y-m-t');
        $siteId = $_GET['site_id'] ?? null;
        $type = $_GET['type'] ?? null;
        
        try {
            $where = "t.date_transaction BETWEEN ? AND ?";
            $params = [$dateDebut, $dateFin];
            
            if ($siteId) {
                $where .= " AND t.site_id = ?";
                $params[] = $siteId;
            }
            
            if ($type) {
                $where .= " AND t.type_transaction = ?";
                $params[] = $type;
            }
            
            $stmt = $this->db->prepare("
                SELECT t.*, s.nom as site_nom, u.nom as utilisateur_nom
                FROM transactions t
                JOIN sites s ON t.site_id = s.id
                JOIN utilisateurs u ON t.utilisateur_id = u.id
                WHERE $where
                ORDER BY t.date_transaction DESC, t.date_creation DESC
                LIMIT 1000
            ");
            $stmt->execute($params);
            $transactions = $stmt->fetchAll();
            
            // Résumé par type
            $stmt = $this->db->prepare("
                SELECT type_transaction, categorie,
                       COUNT(*) as nb_transactions,
                       SUM(montant) as montant_total
                FROM transactions t
                WHERE $where
                GROUP BY type_transaction, categorie
                ORDER BY type_transaction, categorie
            ");
            $stmt->execute($params);
            $resume = $stmt->fetchAll();
            
            ApiResponse::success([
                'periode' => ['debut' => $dateDebut, 'fin' => $dateFin],
                'transactions' => $transactions,
                'resume' => $resume
            ], 'Rapport des transactions généré');
            
        } catch (Exception $e) {
            Utils::logError('Erreur rapport transactions: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du rapport des transactions', 500);
        }
    }
    
    private function getPerformanceReport() {
        $periode = $_GET['periode'] ?? 'mois'; // mois, trimestre, annee
        $siteId = $_GET['site_id'] ?? null;
        
        try {
            $dateFormat = $periode === 'mois' ? '%Y-%m' : ($periode === 'trimestre' ? '%Y-Q%q' : '%Y');
            $dateGroup = $periode === 'mois' ? 'YEAR(date_vente), MONTH(date_vente)' : 
                        ($periode === 'trimestre' ? 'YEAR(date_vente), QUARTER(date_vente)' : 'YEAR(date_vente)');
            
            $where = "1=1";
            $params = [];
            
            if ($siteId) {
                $where .= " AND v.site_id = ?";
                $params[] = $siteId;
            }
            
            // Performance des ventes
            $stmt = $this->db->prepare("
                SELECT DATE_FORMAT(v.date_vente, '$dateFormat') as periode,
                       COUNT(DISTINCT v.client_id) as nb_clients,
                       COUNT(v.id) as nb_ventes,
                       SUM(v.quantite) as quantite_totale,
                       SUM(v.montant_total) as chiffre_affaires,
                       SUM(v.marge_brute) as marge_totale,
                       AVG(v.marge_brute) as marge_moyenne,
                       MIN(v.date_vente) as date_debut,
                       MAX(v.date_vente) as date_fin
                FROM ventes v
                WHERE $where
                GROUP BY $dateGroup
                ORDER BY MIN(v.date_vente) DESC
                LIMIT 24
            ");
            $stmt->execute($params);
            $performanceVentes = $stmt->fetchAll();
            
            // Performance des achats
            $stmt = $this->db->prepare("
                SELECT DATE_FORMAT(a.date_achat, '$dateFormat') as periode,
                       COUNT(DISTINCT a.fournisseur_id) as nb_fournisseurs,
                       COUNT(a.id) as nb_achats,
                       SUM(a.quantite) as quantite_totale,
                       SUM(a.cout_total) as cout_total,
                       AVG(a.prix_unitaire) as prix_moyen
                FROM achats a
                WHERE $where
                GROUP BY $dateGroup
                ORDER BY MIN(a.date_achat) DESC
                LIMIT 24
            ");
            $stmt->execute($params);
            $performanceAchats = $stmt->fetchAll();
            
            ApiResponse::success([
                'periode_type' => $periode,
                'ventes' => $performanceVentes,
                'achats' => $performanceAchats
            ], 'Rapport de performance généré');
            
        } catch (Exception $e) {
            Utils::logError('Erreur rapport performance: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du rapport de performance', 500);
        }
    }
    
    private function getEvolutionReport() {
        $indicateur = $_GET['indicateur'] ?? 'chiffre_affaires'; // chiffre_affaires, marge, roi, volume
        $siteId = $_GET['site_id'] ?? null;
        $produitId = $_GET['produit_id'] ?? null;
        
        try {
            $where = "1=1";
            $params = [];
            
            if ($siteId) {
                $where .= " AND v.site_id = ?";
                $params[] = $siteId;
            }
            
            if ($produitId) {
                $where .= " AND v.produit_id = ?";
                $params[] = $produitId;
            }
            
            // Évolution sur les 12 derniers mois
            $evolution = [];
            for ($i = 11; $i >= 0; $i--) {
                $date = new DateTime();
                $date->modify("-$i months");
                $mois = $date->format('Y-m');
                $startDate = $mois . '-01';
                $endDate = $date->format('Y-m-t');
                
                $stmt = $this->db->prepare("
                    SELECT COUNT(v.id) as nb_ventes,
                           SUM(v.quantite) as volume,
                           SUM(v.montant_total) as chiffre_affaires,
                           SUM(v.marge_brute) as marge,
                           CASE 
                               WHEN SUM(v.cout_achat) > 0 
                               THEN (SUM(v.marge_brute) / SUM(v.cout_achat)) * 100 
                               ELSE 0 
                           END as roi
                    FROM ventes v
                    WHERE $where AND v.date_vente BETWEEN ? AND ?
                ");
                $stmt->execute(array_merge($params, [$startDate, $endDate]));
                $data = $stmt->fetch();
                
                $evolution[] = [
                    'mois' => $mois,
                    'mois_libelle' => $date->format('M Y'),
                    'nb_ventes' => $data['nb_ventes'] ?? 0,
                    'volume' => $data['volume'] ?? 0,
                    'chiffre_affaires' => $data['chiffre_affaires'] ?? 0,
                    'marge' => $data['marge'] ?? 0,
                    'roi' => $data['roi'] ?? 0
                ];
            }
            
            ApiResponse::success([
                'indicateur' => $indicateur,
                'evolution' => $evolution
            ], 'Rapport d\'évolution généré');
            
        } catch (Exception $e) {
            Utils::logError('Erreur rapport évolution: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la génération du rapport d\'évolution', 500);
        }
    }
}

try {
    $api = new ReportingAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API reporting: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>