<?php
	// api/models/Vente.php

class Vente extends BaseModel {
    protected $table = 'ventes';
    protected $fillable = [
        'site_id', 'client_id', 'produit_id', 'quantite', 'prix_unitaire',
        'montant_total', 'cout_achat', 'marge_brute', 'date_vente',
        'numero_facture', 'statut', 'utilisateur_id'
    ];
    
    public function getSite() {
        $stmt = $this->db->prepare("SELECT * FROM sites WHERE id = ?");
        $stmt->execute([$this->site_id]);
        return $stmt->fetch();
    }
    
    public function getClient() {
        $stmt = $this->db->prepare("SELECT * FROM clients WHERE id = ?");
        $stmt->execute([$this->client_id]);
        return $stmt->fetch();
    }
    
    public function getProduit() {
        $stmt = $this->db->prepare("SELECT * FROM produits WHERE id = ?");
        $stmt->execute([$this->produit_id]);
        return $stmt->fetch();
    }
    
    public function updateStock() {
        // Récupérer le stock
        $stmt = $this->db->prepare("SELECT * FROM stocks WHERE site_id = ? AND produit_id = ?");
        $stmt->execute([$this->site_id, $this->produit_id]);
        $stock = $stmt->fetch();
        
        if (!$stock || $stock['quantite_actuelle'] < $this->quantite) {
            throw new Exception("Stock insuffisant pour cette vente");
        }
        
        // Calculer le coût d'achat (FIFO)
        $coutAchat = $this->calculateCoutAchatFIFO($stock['id'], $this->quantite);
        
        // Mettre à jour le stock
        $nouvelleQuantite = $stock['quantite_actuelle'] - $this->quantite;
        $nouvelleValeur = $stock['valeur_stock'] - $coutAchat;
        
        $stmt = $this->db->prepare("
            UPDATE stocks 
            SET quantite_actuelle = ?, 
                valeur_stock = ?,
                derniere_maj = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$nouvelleQuantite, $nouvelleValeur, $stock['id']]);
        
        // Enregistrer le mouvement de stock
        $stmt = $this->db->prepare("
            INSERT INTO mouvements_stock (stock_id, type_mouvement, quantite, prix_unitaire, reference_vente_id)
            VALUES (?, 'sortie', ?, ?, ?)
        ");
        $stmt->execute([$stock['id'], $this->quantite, $this->prix_unitaire, $this->id]);
        
        // Mettre à jour le coût et la marge de la vente
        $this->cout_achat = $coutAchat;
        $this->marge_brute = $this->montant_total - $coutAchat;
        
        $stmt = $this->db->prepare("UPDATE ventes SET cout_achat = ?, marge_brute = ? WHERE id = ?");
        $stmt->execute([$this->cout_achat, $this->marge_brute, $this->id]);
        
        return true;
    }
    
    private function calculateCoutAchatFIFO($stockId, $quantiteVendue) {
        // Récupérer les entrées en stock par ordre chronologique (FIFO)
        $stmt = $this->db->prepare("
            SELECT quantite, prix_unitaire, date_mouvement
            FROM mouvements_stock 
            WHERE stock_id = ? AND type_mouvement = 'entree'
            ORDER BY date_mouvement ASC
        ");
        $stmt->execute([$stockId]);
        $entrees = $stmt->fetchAll();
        
        $coutTotal = 0;
        $quantiteRestante = $quantiteVendue;
        
        foreach ($entrees as $entree) {
            if ($quantiteRestante <= 0) break;
            
            $quantiteAPrendre = min($quantiteRestante, $entree['quantite']);
            $coutTotal += $quantiteAPrendre * $entree['prix_unitaire'];
            $quantiteRestante -= $quantiteAPrendre;
        }
        
        return $coutTotal;
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'site_id' => ['required' => true, 'numeric' => true],
            'client_id' => ['required' => true, 'numeric' => true],
            'produit_id' => ['required' => true, 'numeric' => true],
            'quantite' => ['required' => true, 'numeric' => true, 'positive' => true],
            'prix_unitaire' => ['required' => true, 'numeric' => true, 'positive' => true],
            'date_vente' => ['required' => true, 'date' => true],
            'statut' => ['in_array' => ['en_cours', 'livre', 'paye']]
        ];
        
        return Validator::validate($data, $rules);
    }
    
    public function calculateTotals() {
        $this->montant_total = $this->quantite * $this->prix_unitaire;
    }
}
?>