<?php
// api/models/Site.php

class Site extends BaseModel {
    protected $table = 'sites';
    protected $fillable = [
        'nom', 'adresse', 'type_site', 'coordonnees_gps', 
        'responsable_id', 'actif'
    ];
    
    public function getBudgets() {
        $stmt = $this->db->prepare("SELECT * FROM budgets WHERE site_id = ? ORDER BY date_creation DESC");
        $stmt->execute([$this->id]);
        return $stmt->fetchAll();
    }
    
    public function getActiveBudget() {
        $stmt = $this->db->prepare("SELECT * FROM budgets WHERE site_id = ? AND statut = 'actif' ORDER BY date_creation DESC LIMIT 1");
        $stmt->execute([$this->id]);
        return $stmt->fetch();
    }
    
    public function getTotalBudget() {
        $stmt = $this->db->prepare("SELECT SUM(montant_actuel) as total FROM budgets WHERE site_id = ? AND statut = 'actif'");
        $stmt->execute([$this->id]);
        $result = $stmt->fetch();
        return $result['total'] ?? 0;
    }
    
    public function getMonthlyStats($year, $month) {
        $startDate = "$year-$month-01";
        $endDate = date('Y-m-t', strtotime($startDate));
        
        // Achats du mois
        $stmt = $this->db->prepare("
            SELECT COALESCE(SUM(cout_total), 0) as total_achats, COUNT(*) as nb_achats
            FROM achats 
            WHERE site_id = ? AND date_achat BETWEEN ? AND ?
        ");
        $stmt->execute([$this->id, $startDate, $endDate]);
        $achats = $stmt->fetch();
        
        // Ventes du mois
        $stmt = $this->db->prepare("
            SELECT COALESCE(SUM(montant_total), 0) as total_ventes, 
                   COALESCE(SUM(marge_brute), 0) as total_marge,
                   COUNT(*) as nb_ventes
            FROM ventes 
            WHERE site_id = ? AND date_vente BETWEEN ? AND ?
        ");
        $stmt->execute([$this->id, $startDate, $endDate]);
        $ventes = $stmt->fetch();
        
        return [
            'achats' => $achats,
            'ventes' => $ventes,
            'roi' => $achats['total_achats'] > 0 ? ($ventes['total_marge'] / $achats['total_achats']) * 100 : 0
        ];
    }
    
    public function getStocks() {
        $stmt = $this->db->prepare("
            SELECT s.*, p.nom as produit_nom, p.unite_mesure
            FROM stocks s
            JOIN produits p ON s.produit_id = p.id
            WHERE s.site_id = ? AND s.quantite_actuelle > 0
            ORDER BY p.nom
        ");
        $stmt->execute([$this->id]);
        return $stmt->fetchAll();
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'nom' => ['required' => true, 'max_length' => 100],
            'type_site' => ['required' => true, 'in_array' => ['pont_bascule', 'magasin_stockage', 'autre']],
            'adresse' => ['max_length' => 500]
        ];
        
        return Validator::validate($data, $rules);
    }
}

// api/models/Budget.php

class Budget extends BaseModel {
    protected $table = 'budgets';
    protected $fillable = [
        'site_id', 'montant_initial', 'montant_actuel', 'devise',
        'periode_debut', 'periode_fin', 'statut'
    ];
    
    public function getSite() {
        $stmt = $this->db->prepare("SELECT * FROM sites WHERE id = ?");
        $stmt->execute([$this->site_id]);
        return $stmt->fetch();
    }
    
    public function getUsagePercentage() {
        if ($this->montant_initial <= 0) return 0;
        return (($this->montant_initial - $this->montant_actuel) / $this->montant_initial) * 100;
    }
    
    public function updateAmount($amount, $operation = 'subtract') {
        $newAmount = $operation === 'subtract' 
            ? $this->montant_actuel - $amount 
            : $this->montant_actuel + $amount;
            
        $stmt = $this->db->prepare("UPDATE budgets SET montant_actuel = ?, last_modified = NOW() WHERE id = ?");
        return $stmt->execute([$newAmount, $this->id]);
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'site_id' => ['required' => true, 'numeric' => true],
            'montant_initial' => ['required' => true, 'numeric' => true, 'positive' => true],
            'montant_actuel' => ['numeric' => true, 'positive' => true],
            'devise' => ['max_length' => 3],
            'periode_debut' => ['required' => true, 'date' => true],
            'periode_fin' => ['required' => true, 'date' => true],
            'statut' => ['in_array' => ['actif', 'suspendu', 'cloture']]
        ];
        
        return Validator::validate($data, $rules);
    }
}

// api/models/Produit.php

class Produit extends BaseModel {
    protected $table = 'produits';
    protected $fillable = [
        'nom', 'unite_mesure', 'categorie', 'description', 'actif'
    ];
    
    public function getStocksBySite() {
        $stmt = $this->db->prepare("
            SELECT s.*, si.nom as site_nom
            FROM stocks s
            JOIN sites si ON s.site_id = si.id
            WHERE s.produit_id = ? AND s.quantite_actuelle > 0
            ORDER BY si.nom
        ");
        $stmt->execute([$this->id]);
        return $stmt->fetchAll();
    }
    
    public function getTotalStock() {
        $stmt = $this->db->prepare("SELECT SUM(quantite_actuelle) as total FROM stocks WHERE produit_id = ?");
        $stmt->execute([$this->id]);
        $result = $stmt->fetch();
        return $result['total'] ?? 0;
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'nom' => ['required' => true, 'max_length' => 100],
            'unite_mesure' => ['required' => true, 'in_array' => ['kg', 'tonne', 'sac', 'litre']],
            'categorie' => ['required' => true, 'in_array' => ['cereale', 'legumineuse', 'tubercule', 'autre']],
            'description' => ['max_length' => 500]
        ];
        
        return Validator::validate($data, $rules);
    }
}

// api/models/Achat.php

class Achat extends BaseModel {
    protected $table = 'achats';
    protected $fillable = [
        'site_id', 'fournisseur_id', 'produit_id', 'quantite', 'prix_unitaire',
        'montant_total', 'frais_transport', 'autres_frais', 'cout_total',
        'date_achat', 'numero_bon', 'statut', 'utilisateur_id'
    ];
    
    public function getSite() {
        $stmt = $this->db->prepare("SELECT * FROM sites WHERE id = ?");
        $stmt->execute([$this->site_id]);
        return $stmt->fetch();
    }
    
    public function getFournisseur() {
        $stmt = $this->db->prepare("SELECT * FROM fournisseurs WHERE id = ?");
        $stmt->execute([$this->fournisseur_id]);
        return $stmt->fetch();
    }
    
    public function getProduit() {
        $stmt = $this->db->prepare("SELECT * FROM produits WHERE id = ?");
        $stmt->execute([$this->produit_id]);
        return $stmt->fetch();
    }
    
    public function updateStock() {
        // Vérifier si le stock existe
        $stmt = $this->db->prepare("SELECT * FROM stocks WHERE site_id = ? AND produit_id = ?");
        $stmt->execute([$this->site_id, $this->produit_id]);
        $stock = $stmt->fetch();
        
        if ($stock) {
            // Mettre à jour le stock existant (CMUP)
            $nouvelleQuantite = $stock['quantite_actuelle'] + $this->quantite;
            $nouvelleValeur = $stock['valeur_stock'] + $this->cout_total;
            $nouveauPrix = $nouvelleQuantite > 0 ? $nouvelleValeur / $nouvelleQuantite : 0;
            
            $stmt = $this->db->prepare("
                UPDATE stocks 
                SET quantite_actuelle = ?, 
                    valeur_stock = ?, 
                    prix_moyen_pondere = ?,
                    derniere_maj = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$nouvelleQuantite, $nouvelleValeur, $nouveauPrix, $stock['id']]);
            $stockId = $stock['id'];
        } else {
            // Créer un nouveau stock
            $stmt = $this->db->prepare("
                INSERT INTO stocks (site_id, produit_id, quantite_actuelle, valeur_stock, prix_moyen_pondere)
                VALUES (?, ?, ?, ?, ?)
            ");
            $prixMoyen = $this->quantite > 0 ? $this->cout_total / $this->quantite : 0;
            $stmt->execute([$this->site_id, $this->produit_id, $this->quantite, $this->cout_total, $prixMoyen]);
            $stockId = $this->db->lastInsertId();
        }
        
        // Enregistrer le mouvement de stock
        $stmt = $this->db->prepare("
            INSERT INTO mouvements_stock (stock_id, type_mouvement, quantite, prix_unitaire, reference_achat_id)
            VALUES (?, 'entree', ?, ?, ?)
        ");
        $stmt->execute([$stockId, $this->quantite, $this->prix_unitaire, $this->id]);
        
        return true;
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'site_id' => ['required' => true, 'numeric' => true],
            'fournisseur_id' => ['required' => true, 'numeric' => true],
            'produit_id' => ['required' => true, 'numeric' => true],
            'quantite' => ['required' => true, 'numeric' => true, 'positive' => true],
            'prix_unitaire' => ['required' => true, 'numeric' => true, 'positive' => true],
            'date_achat' => ['required' => true, 'date' => true],
            'statut' => ['in_array' => ['en_cours', 'livre', 'paye']]
        ];
        
        return Validator::validate($data, $rules);
    }
    
    public function calculateTotals() {
        $this->montant_total = $this->quantite * $this->prix_unitaire;
        $this->cout_total = $this->montant_total + ($this->frais_transport ?? 0) + ($this->autres_frais ?? 0);
    }
}

// api/models/Vente.php

class Vente extends BaseModel {
    protected $table = 'ventes';
    protected $fillable = [
        'site_id', 'client_id', 'produit_id', 'quantite', 'prix_unitaire',
        'montant_total', 'cout_achat', 'marge_brute', 'date_vente',
        'numero_facture', 'statut', 'utilisateur_id'
    ];
    
    public function getSite() {
        $stmt = $this->db->prepare("SELECT * FROM sites WHERE id = ?");
        $stmt->execute([$this->site_id]);
        return $stmt->fetch();
    }
    
    public function getClient() {
        $stmt = $this->db->prepare("SELECT * FROM clients WHERE id = ?");
        $stmt->execute([$this->client_id]);
        return $stmt->fetch();
    }
    
    public function getProduit() {
        $stmt = $this->db->prepare("SELECT * FROM produits WHERE id = ?");
        $stmt->execute([$this->produit_id]);
        return $stmt->fetch();
    }
    
    public function updateStock() {
        // Récupérer le stock
        $stmt = $this->db->prepare("SELECT * FROM stocks WHERE site_id = ? AND produit_id = ?");
        $stmt->execute([$this->site_id, $this->produit_id]);
        $stock = $stmt->fetch();
        
        if (!$stock || $stock['quantite_actuelle'] < $this->quantite) {
            throw new Exception("Stock insuffisant pour cette vente");
        }
        
        // Calculer le coût d'achat (FIFO)
        $coutAchat = $this->calculateCoutAchatFIFO($stock['id'], $this->quantite);
        
        // Mettre à jour le stock
        $nouvelleQuantite = $stock['quantite_actuelle'] - $this->quantite;
        $nouvelleValeur = $stock['valeur_stock'] - $coutAchat;
        
        $stmt = $this->db->prepare("
            UPDATE stocks 
            SET quantite_actuelle = ?, 
                valeur_stock = ?,
                derniere_maj = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$nouvelleQuantite, $nouvelleValeur, $stock['id']]);
        
        // Enregistrer le mouvement de stock
        $stmt = $this->db->prepare("
            INSERT INTO mouvements_stock (stock_id, type_mouvement, quantite, prix_unitaire, reference_vente_id)
            VALUES (?, 'sortie', ?, ?, ?)
        ");
        $stmt->execute([$stock['id'], $this->quantite, $this->prix_unitaire, $this->id]);
        
        // Mettre à jour le coût et la marge de la vente
        $this->cout_achat = $coutAchat;
        $this->marge_brute = $this->montant_total - $coutAchat;
        
        $stmt = $this->db->prepare("UPDATE ventes SET cout_achat = ?, marge_brute = ? WHERE id = ?");
        $stmt->execute([$this->cout_achat, $this->marge_brute, $this->id]);
        
        return true;
    }
    
    private function calculateCoutAchatFIFO($stockId, $quantiteVendue) {
        // Récupérer les entrées en stock par ordre chronologique (FIFO)
        $stmt = $this->db->prepare("
            SELECT quantite, prix_unitaire, date_mouvement
            FROM mouvements_stock 
            WHERE stock_id = ? AND type_mouvement = 'entree'
            ORDER BY date_mouvement ASC
        ");
        $stmt->execute([$stockId]);
        $entrees = $stmt->fetchAll();
        
        $coutTotal = 0;
        $quantiteRestante = $quantiteVendue;
        
        foreach ($entrees as $entree) {
            if ($quantiteRestante <= 0) break;
            
            $quantiteAPrendre = min($quantiteRestante, $entree['quantite']);
            $coutTotal += $quantiteAPrendre * $entree['prix_unitaire'];
            $quantiteRestante -= $quantiteAPrendre;
        }
        
        return $coutTotal;
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'site_id' => ['required' => true, 'numeric' => true],
            'client_id' => ['required' => true, 'numeric' => true],
            'produit_id' => ['required' => true, 'numeric' => true],
            'quantite' => ['required' => true, 'numeric' => true, 'positive' => true],
            'prix_unitaire' => ['required' => true, 'numeric' => true, 'positive' => true],
            'date_vente' => ['required' => true, 'date' => true],
            'statut' => ['in_array' => ['en_cours', 'livre', 'paye']]
        ];
        
        return Validator::validate($data, $rules);
    }
    
    public function calculateTotals() {
        $this->montant_total = $this->quantite * $this->prix_unitaire;
    }
}

// api/models/Fournisseur.php

class Fournisseur extends BaseModel {
    protected $table = 'fournisseurs';
    protected $fillable = [
        'nom', 'telephone', 'adresse', 'type_produit', 'actif'
    ];
    
    public function getAchats($limit = null) {
        $sql = "SELECT * FROM achats WHERE fournisseur_id = ? ORDER BY date_achat DESC";
        if ($limit) {
            $sql .= " LIMIT $limit";
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$this->id]);
        return $stmt->fetchAll();
    }
    
    public function getTotalAchats($year = null) {
        $sql = "SELECT COALESCE(SUM(cout_total), 0) as total FROM achats WHERE fournisseur_id = ?";
        $params = [$this->id];
        
        if ($year) {
            $sql .= " AND YEAR(date_achat) = ?";
            $params[] = $year;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'nom' => ['required' => true, 'max_length' => 100],
            'telephone' => ['max_length' => 20],
            'adresse' => ['max_length' => 500],
            'type_produit' => ['max_length' => 50]
        ];
        
        return Validator::validate($data, $rules);
    }
}

// api/models/Client.php

class Client extends BaseModel {
    protected $table = 'clients';
    protected $fillable = [
        'nom', 'telephone', 'adresse', 'type_client', 'limite_credit', 'actif'
    ];
    
    public function getVentes($limit = null) {
        $sql = "SELECT * FROM ventes WHERE client_id = ? ORDER BY date_vente DESC";
        if ($limit) {
            $sql .= " LIMIT $limit";
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$this->id]);
        return $stmt->fetchAll();
    }
    
    public function getTotalVentes($year = null) {
        $sql = "SELECT COALESCE(SUM(montant_total), 0) as total FROM ventes WHERE client_id = ?";
        $params = [$this->id];
        
        if ($year) {
            $sql .= " AND YEAR(date_vente) = ?";
            $params[] = $year;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    public function validate($data, $isUpdate = false) {
        $rules = [
            'nom' => ['required' => true, 'max_length' => 100],
            'telephone' => ['max_length' => 20],
            'adresse' => ['max_length' => 500],
            'type_client' => ['required' => true, 'in_array' => ['grossiste', 'detaillant', 'particulier']],
            'limite_credit' => ['numeric' => true, 'positive' => true]
        ];
        
        return Validator::validate($data, $rules);
    }
}

?>