<?php
// api/fournisseurs.php - Endpoint pour la gestion des fournisseurs

require_once 'config.php';

class FournisseursAPI {
    private $fournisseur;
    private $user;
    
    public function __construct() {
        $this->fournisseur = new Fournisseur();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    if (isset($pathParts[3]) && $pathParts[3] === 'achats') {
                        $this->getFournisseurAchats($id);
                    } else {
                        $this->getFournisseur($id);
                    }
                } else {
                    $this->getFournisseurs();
                }
                break;
                
            case 'POST':
                $this->createFournisseur();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateFournisseur($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteFournisseur($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getFournisseurs() {
        $where = [];
        
        if (isset($_GET['actif'])) {
            $where['actif'] = $_GET['actif'] === '1' ? 1 : 0;
        }
        
        $fournisseurs = $this->fournisseur->findAll($where, 'nom ASC');
        
        // Enrichir avec les statistiques
        foreach ($fournisseurs as &$fournisseur) {
            $fournisseurObj = new Fournisseur();
            $fournisseurObj->id = $fournisseur['id'];
            $fournisseur['total_achats_annee'] = $fournisseurObj->getTotalAchats(date('Y'));
            $fournisseur['nb_achats'] = count($fournisseurObj->getAchats(10));
        }
        
        ApiResponse::success($fournisseurs, 'Fournisseurs récupérés');
    }
    
    private function getFournisseur($id) {
        $fournisseur = $this->fournisseur->find($id);
        
        if (!$fournisseur) {
            ApiResponse::error('Fournisseur non trouvé', 404);
        }
        
        // Enrichir avec les données détaillées
        $fournisseurObj = new Fournisseur();
        $fournisseurObj->id = $id;
        $fournisseur['achats_recents'] = $fournisseurObj->getAchats(20);
        $fournisseur['total_achats_annee'] = $fournisseurObj->getTotalAchats(date('Y'));
        $fournisseur['total_achats_global'] = $fournisseurObj->getTotalAchats();
        
        ApiResponse::success($fournisseur, 'Fournisseur récupéré');
    }
    
    private function getFournisseurAchats($id) {
        $fournisseur = $this->fournisseur->find($id);
        if (!$fournisseur) {
            ApiResponse::error('Fournisseur non trouvé', 404);
        }
        
        $fournisseurObj = new Fournisseur();
        $fournisseurObj->id = $id;
        $achats = $fournisseurObj->getAchats($_GET['limit'] ?? 50);
        
        ApiResponse::success($achats, 'Achats du fournisseur récupérés');
    }
    
    private function createFournisseur() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->fournisseur->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $id = $this->fournisseur->create($data);
            ApiResponse::success(['id' => $id], 'Fournisseur créé avec succès', 201);
            
        } catch (Exception $e) {
            Utils::logError('Erreur création fournisseur: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création du fournisseur', 500);
        }
    }
    
    private function updateFournisseur($id) {
        $fournisseur = $this->fournisseur->find($id);
        if (!$fournisseur) {
            ApiResponse::error('Fournisseur non trouvé', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->fournisseur->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->fournisseur->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Fournisseur mis à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour fournisseur: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour du fournisseur', 500);
        }
    }
    
    private function deleteFournisseur($id) {
        $fournisseur = $this->fournisseur->find($id);
        if (!$fournisseur) {
            ApiResponse::error('Fournisseur non trouvé', 404);
        }
        
        // Vérifier s'il y a des achats associés
        $fournisseurObj = new Fournisseur();
        $fournisseurObj->id = $id;
        $achats = $fournisseurObj->getAchats(1);
        
        if (!empty($achats)) {
            ApiResponse::error('Impossible de supprimer un fournisseur avec des achats associés', 400);
        }
        
        try {
            $success = $this->fournisseur->delete($id);
            
            if ($success) {
                ApiResponse::success(null, 'Fournisseur supprimé avec succès');
            } else {
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur suppression fournisseur: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression du fournisseur', 500);
        }
    }
}

try {
    $api = new FournisseursAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API fournisseurs: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>