<?php
// api/config.php - Configuration de base pour l'API

// Configuration de la base de données
define('DB_HOST', 'serverkmaf.helioho.st');
define('DB_NAME', 'kombarferd_gestfinancedb');
define('DB_USER', 'kombarferd_daf');
define('DB_PASS', 'DnK^2d0?yOpfidu5');
define('DB_CHARSET', 'utf8mb4');

// Configuration de l'application
define('APP_ENV', 'development'); // production, development
define('JWT_SECRET', '9mq8lKl2lPAevijFuJZ1Q0dSqIBNH1JRlarA25eME06TpVBVSc3urjeDzgr8mEMV');
define('API_VERSION', '1.0.0');

// Configuration des uploads
define('UPLOAD_MAX_SIZE', 10 * 1024 * 1024); // 10MB
define('UPLOAD_PATH', __DIR__ . '/../uploads/');

// Gestion des erreurs
error_reporting(APP_ENV === 'development' ? E_ALL : 0);
ini_set('display_errors', APP_ENV === 'development' ? 1 : 0);

// Headers CORS
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Répondre aux requêtes OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

/**
 * Classe de connexion à la base de données
 */
class Database {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ]);
        } catch (PDOException $e) {
            error_log("Erreur de connexion à la base de données: " . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Erreur de connexion à la base de données']);
            exit();
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    // Empêcher le clonage
    private function __clone() {}
    
    // Empêcher la désérialisation
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

/**
 * Classe utilitaire pour les réponses API
 */
class ApiResponse {
    
    public static function success($data = null, $message = null, $code = 200) {
        http_response_code($code);
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c')
        ]);
        exit();
    }
    
    public static function error($message, $code = 400, $details = null) {
        http_response_code($code);
        echo json_encode([
            'success' => false,
            'error' => $message,
            'details' => $details,
            'timestamp' => date('c')
        ]);
        exit();
    }
    
    public static function validation($errors) {
        http_response_code(422);
        echo json_encode([
            'success' => false,
            'error' => 'Erreurs de validation',
            'validation_errors' => $errors,
            'timestamp' => date('c')
        ]);
        exit();
    }
}

/**
 * Classe de validation des données
 */
class Validator {
    
    public static function required($value, $field) {
        if (empty($value) && $value !== 0 && $value !== '0') {
            return "Le champ $field est requis";
        }
        return null;
    }
    
    public static function email($value, $field) {
        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
            return "Le champ $field doit être une adresse email valide";
        }
        return null;
    }
    
    public static function numeric($value, $field) {
        if (!empty($value) && !is_numeric($value)) {
            return "Le champ $field doit être numérique";
        }
        return null;
    }
    
    public static function positive($value, $field) {
        if (!empty($value) && $value < 0) {
            return "Le champ $field doit être positif";
        }
        return null;
    }
    
    public static function minLength($value, $field, $min) {
        if (!empty($value) && strlen($value) < $min) {
            return "Le champ $field doit contenir au moins $min caractères";
        }
        return null;
    }
    
    public static function maxLength($value, $field, $max) {
        if (!empty($value) && strlen($value) > $max) {
            return "Le champ $field ne peut pas dépasser $max caractères";
        }
        return null;
    }
    
    public static function inArray($value, $field, $allowed) {
        if (!empty($value) && !in_array($value, $allowed)) {
            return "Le champ $field doit être l'une des valeurs suivantes: " . implode(', ', $allowed);
        }
        return null;
    }
    
    public static function date($value, $field) {
        if (!empty($value)) {
            $d = DateTime::createFromFormat('Y-m-d', $value);
            if (!$d || $d->format('Y-m-d') !== $value) {
                return "Le champ $field doit être une date valide (YYYY-MM-DD)";
            }
        }
        return null;
    }
    
    public static function validate($data, $rules) {
        $errors = [];
        
        foreach ($rules as $field => $fieldRules) {
            $value = $data[$field] ?? null;
            
            foreach ($fieldRules as $rule => $params) {
                $error = null;
                
                switch ($rule) {
                    case 'required':
                        $error = self::required($value, $field);
                        break;
                    case 'email':
                        $error = self::email($value, $field);
                        break;
                    case 'numeric':
                        $error = self::numeric($value, $field);
                        break;
                    case 'positive':
                        $error = self::positive($value, $field);
                        break;
                    case 'min_length':
                        $error = self::minLength($value, $field, $params);
                        break;
                    case 'max_length':
                        $error = self::maxLength($value, $field, $params);
                        break;
                    case 'in_array':
                        $error = self::inArray($value, $field, $params);
                        break;
                    case 'date':
                        $error = self::date($value, $field);
                        break;
                }
                
                if ($error) {
                    $errors[$field] = $error;
                    break; // Arrêter à la première erreur pour ce champ
                }
            }
        }
        
        return $errors;
    }
}

/**
 * Classe pour la gestion de l'authentification JWT
 */
class Auth {
    
    public static function generateToken($userId, $userData = []) {
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        
        $payload = json_encode([
            'user_id' => $userId,
            'user_data' => $userData,
            'iat' => time(),
            'exp' => time() + (24 * 60 * 60) // 24 heures
        ]);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, JWT_SECRET, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }
    
    public static function verifyToken($token) {
        $tokenParts = explode('.', $token);
        
        if (count($tokenParts) !== 3) {
            return false;
        }
        
        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[1]));
        $signatureProvided = $tokenParts[2];
        
        $expectedSignature = hash_hmac('sha256', $tokenParts[0] . "." . $tokenParts[1], JWT_SECRET, true);
        $base64ExpectedSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($expectedSignature));
        
        if (!hash_equals($base64ExpectedSignature, $signatureProvided)) {
            return false;
        }
        
        $payloadData = json_decode($payload, true);
        
        if ($payloadData['exp'] < time()) {
            return false; // Token expiré
        }
        
        return $payloadData;
    }
    
    public static function getCurrentUser() {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? null;
        
        if (!$authHeader) {
            return null;
        }
        
        if (!preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            return null;
        }
        
        $token = $matches[1];
        return self::verifyToken($token);
    }
    
    public static function requireAuth() {
        $user = self::getCurrentUser();
        
        if (!$user) {
            ApiResponse::error('Authentification requise', 401);
        }
        
        return $user;
    }
}

/**
 * Classe de base pour les modèles
 */
abstract class BaseModel {
    protected $db;
    protected $table;
    protected $primaryKey = 'id';
    protected $fillable = [];
    protected $timestamps = true;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    public function find($id) {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE {$this->primaryKey} = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    public function findAll($where = [], $orderBy = null, $limit = null) {
        $sql = "SELECT * FROM {$this->table}";
        $params = [];
        
        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $field => $value) {
                $conditions[] = "$field = ?";
                $params[] = $value;
            }
            $sql .= " WHERE " . implode(' AND ', $conditions);
        }
        
        if ($orderBy) {
            $sql .= " ORDER BY $orderBy";
        }
        
        if ($limit) {
            $sql .= " LIMIT $limit";
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }
    
    public function create($data) {
        $data = $this->filterFillable($data);
        
        if ($this->timestamps) {
            $data['date_creation'] = date('Y-m-d H:i:s');
            $data['last_modified'] = date('Y-m-d H:i:s');
        }
        
        $fields = array_keys($data);
        $placeholders = array_fill(0, count($fields), '?');
        
        $sql = "INSERT INTO {$this->table} (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $placeholders) . ")";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(array_values($data));
        
        return $this->db->lastInsertId();
    }
    
    public function update($id, $data) {
        $data = $this->filterFillable($data);
        
        if ($this->timestamps) {
            $data['last_modified'] = date('Y-m-d H:i:s');
        }
        
        $fields = array_keys($data);
        $setClause = implode(' = ?, ', $fields) . ' = ?';
        
        $sql = "UPDATE {$this->table} SET $setClause WHERE {$this->primaryKey} = ?";
        
        $params = array_values($data);
        $params[] = $id;
        
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($params);
    }
    
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM {$this->table} WHERE {$this->primaryKey} = ?");
        return $stmt->execute([$id]);
    }
    
    protected function filterFillable($data) {
        if (empty($this->fillable)) {
            return $data;
        }
        
        return array_intersect_key($data, array_flip($this->fillable));
    }
    
    public function beginTransaction() {
        return $this->db->beginTransaction();
    }
    
    public function commit() {
        return $this->db->commit();
    }
    
    public function rollback() {
        return $this->db->rollback();
    }
}

/**
 * Utilitaires divers
 */
class Utils {
    
    public static function sanitizeInput($data) {
        if (is_array($data)) {
            return array_map([self::class, 'sanitizeInput'], $data);
        }
        
        return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
    }
    
    public static function generateUUID() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    public static function logError($message, $context = []) {
        $logMessage = date('Y-m-d H:i:s') . " - " . $message;
        if (!empty($context)) {
            $logMessage .= " - Context: " . json_encode($context);
        }
        error_log($logMessage . PHP_EOL, 3, __DIR__ . '/../logs/api_errors.log');
    }
    
    public static function formatCurrency($amount, $currency = 'XOF') {
        return number_format($amount, 0, ',', ' ') . ' ' . $currency;
    }
    
    public static function parseJsonInput() {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            ApiResponse::error('Format JSON invalide', 400);
        }
        
        return $data ?: [];
    }
}

// Auto-chargement des modèles
spl_autoload_register(function ($className) {
    $file = __DIR__ . '/models/' . $className . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

// Définir le timezone
date_default_timezone_set('Africa/Abidjan');

// Log de la requête API pour debug
if (APP_ENV === 'development') {
    Utils::logError('API Request: ' . $_SERVER['REQUEST_METHOD'] . ' ' . $_SERVER['REQUEST_URI'], [
        'headers' => getallheaders(),
        'body' => file_get_contents('php://input')
    ]);
}

?>