<?php
// api/clients.php - Endpoint pour la gestion des clients

require_once 'config.php';

class ClientsAPI {
    private $client;
    private $user;
    
    public function __construct() {
        $this->client = new Client();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    if (isset($pathParts[3]) && $pathParts[3] === 'ventes') {
                        $this->getClientVentes($id);
                    } else {
                        $this->getClient($id);
                    }
                } else {
                    $this->getClients();
                }
                break;
                
            case 'POST':
                $this->createClient();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateClient($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteClient($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getClients() {
        $where = [];
        
        if (isset($_GET['actif'])) {
            $where['actif'] = $_GET['actif'] === '1' ? 1 : 0;
        }
        
        if (isset($_GET['type_client'])) {
            $where['type_client'] = $_GET['type_client'];
        }
        
        $clients = $this->client->findAll($where, 'nom ASC');
        
        // Enrichir avec les statistiques
        foreach ($clients as &$client) {
            $clientObj = new Client();
            $clientObj->id = $client['id'];
            $client['total_ventes_annee'] = $clientObj->getTotalVentes(date('Y'));
            $client['nb_ventes'] = count($clientObj->getVentes(10));
        }
        
        ApiResponse::success($clients, 'Clients récupérés');
    }
    
    private function getClient($id) {
        $client = $this->client->find($id);
        
        if (!$client) {
            ApiResponse::error('Client non trouvé', 404);
        }
        
        // Enrichir avec les données détaillées
        $clientObj = new Client();
        $clientObj->id = $id;
        $client['ventes_recentes'] = $clientObj->getVentes(20);
        $client['total_ventes_annee'] = $clientObj->getTotalVentes(date('Y'));
        $client['total_ventes_global'] = $clientObj->getTotalVentes();
        
        ApiResponse::success($client, 'Client récupéré');
    }
    
    private function getClientVentes($id) {
        $client = $this->client->find($id);
        if (!$client) {
            ApiResponse::error('Client non trouvé', 404);
        }
        
        $clientObj = new Client();
        $clientObj->id = $id;
        $ventes = $clientObj->getVentes($_GET['limit'] ?? 50);
        
        ApiResponse::success($ventes, 'Ventes du client récupérées');
    }
    
    private function createClient() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->client->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $id = $this->client->create($data);
            ApiResponse::success(['id' => $id], 'Client créé avec succès', 201);
            
        } catch (Exception $e) {
            Utils::logError('Erreur création client: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création du client', 500);
        }
    }
    
    private function updateClient($id) {
        $client = $this->client->find($id);
        if (!$client) {
            ApiResponse::error('Client non trouvé', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->client->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->client->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Client mis à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour client: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour du client', 500);
        }
    }
    
    private function deleteClient($id) {
        $client = $this->client->find($id);
        if (!$client) {
            ApiResponse::error('Client non trouvé', 404);
        }
        
        // Vérifier s'il y a des ventes associées
        $clientObj = new Client();
        $clientObj->id = $id;
        $ventes = $clientObj->getVentes(1);
        
        if (!empty($ventes)) {
            ApiResponse::error('Impossible de supprimer un client avec des ventes associées', 400);
        }
        
        try {
            $success = $this->client->delete($id);
            
            if ($success) {
                ApiResponse::success(null, 'Client supprimé avec succès');
            } else {
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur suppression client: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression du client', 500);
        }
    }
}

try {
    $api = new ClientsAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API clients: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>