<?php
// api/auth.php - API d'authentification

require_once 'config.php';

class AuthAPI {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $action = basename($path);
        
        switch ($method) {
            case 'POST':
                switch ($action) {
                    case 'login':
                        $this->login();
                        break;
                    case 'register':
                        $this->register();
                        break;
                    case 'refresh':
                        $this->refreshToken();
                        break;
                    case 'logout':
                        $this->logout();
                        break;
                    default:
                        ApiResponse::error('Action non reconnue', 404);
                }
                break;
                
            case 'GET':
                switch ($action) {
                    case 'me':
                        $this->getCurrentUser();
                        break;
                    case 'verify':
                        $this->verifyToken();
                        break;
                    default:
                        ApiResponse::error('Action non reconnue', 404);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function login() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = Validator::validate($data, [
            'email' => ['required' => true, 'email' => true],
            'mot_de_passe' => ['required' => true]
        ]);
        
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $email = Utils::sanitizeInput($data['email']);
        $password = $data['mot_de_passe'];
        
        try {
            // Rechercher l'utilisateur
            $stmt = $this->db->prepare("SELECT * FROM utilisateurs WHERE email = ? AND actif = 1");
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if (!$user || !password_verify($password, $user['mot_de_passe'])) {
                // Attendre un peu pour éviter les attaques de timing
                usleep(500000); // 0.5 seconde
                ApiResponse::error('Email ou mot de passe incorrect', 401);
            }
            
            // Mettre à jour la dernière connexion
            $stmt = $this->db->prepare("UPDATE utilisateurs SET derniere_connexion = NOW() WHERE id = ?");
            $stmt->execute([$user['id']]);
            
            // Générer le token JWT
            $userData = [
                'nom' => $user['nom'],
                'email' => $user['email'],
                'role' => $user['role']
            ];
            
            $token = Auth::generateToken($user['id'], $userData);
            
            // Réponse de succès
            ApiResponse::success([
                'token' => $token,
                'user' => [
                    'id' => $user['id'],
                    'nom' => $user['nom'],
                    'email' => $user['email'],
                    'role' => $user['role']
                ]
            ], 'Connexion réussie');
            
        } catch (Exception $e) {
            Utils::logError('Erreur lors de la connexion: ' . $e->getMessage(), ['email' => $email]);
            ApiResponse::error('Erreur lors de la connexion', 500);
        }
    }
    
    private function register() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = Validator::validate($data, [
            'nom' => ['required' => true, 'max_length' => 100],
            'email' => ['required' => true, 'email' => true],
            'mot_de_passe' => ['required' => true, 'min_length' => 6],
            'role' => ['in_array' => ['admin', 'gestionnaire', 'consultant']]
        ]);
        
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $nom = Utils::sanitizeInput($data['nom']);
        $email = Utils::sanitizeInput($data['email']);
        $password = $data['mot_de_passe'];
        $role = $data['role'] ?? 'consultant';
        
        try {
            // Vérifier si l'email existe déjà
            $stmt = $this->db->prepare("SELECT id FROM utilisateurs WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->fetch()) {
                ApiResponse::error('Cet email est déjà utilisé', 409);
            }
            
            // Hacher le mot de passe
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            
            // Créer l'utilisateur
            $stmt = $this->db->prepare("
                INSERT INTO utilisateurs (nom, email, mot_de_passe, role) 
                VALUES (?, ?, ?, ?)
            ");
            
            $stmt->execute([$nom, $email, $hashedPassword, $role]);
            $userId = $this->db->lastInsertId();
            
            // Générer le token
            $userData = [
                'nom' => $nom,
                'email' => $email,
                'role' => $role
            ];
            
            $token = Auth::generateToken($userId, $userData);
            
            ApiResponse::success([
                'token' => $token,
                'user' => [
                    'id' => $userId,
                    'nom' => $nom,
                    'email' => $email,
                    'role' => $role
                ]
            ], 'Compte créé avec succès', 201);
            
        } catch (Exception $e) {
            Utils::logError('Erreur lors de l\'inscription: ' . $e->getMessage(), ['email' => $email]);
            ApiResponse::error('Erreur lors de la création du compte', 500);
        }
    }
    
    private function getCurrentUser() {
        $user = Auth::requireAuth();
        
        try {
            // Récupérer les informations complètes de l'utilisateur
            $stmt = $this->db->prepare("SELECT id, nom, email, role, date_creation, derniere_connexion FROM utilisateurs WHERE id = ?");
            $stmt->execute([$user['user_id']]);
            $userData = $stmt->fetch();
            
            if (!$userData) {
                ApiResponse::error('Utilisateur non trouvé', 404);
            }
            
            ApiResponse::success($userData, 'Informations utilisateur récupérées');
            
        } catch (Exception $e) {
            Utils::logError('Erreur lors de la récupération de l\'utilisateur: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la récupération des informations', 500);
        }
    }
    
    private function verifyToken() {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? null;
        
        if (!$authHeader) {
            ApiResponse::error('Token d\'authentification manquant', 401);
        }
        
        if (!preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            ApiResponse::error('Format de token invalide', 401);
        }
        
        $token = $matches[1];
        $payload = Auth::verifyToken($token);
        
        if (!$payload) {
            ApiResponse::error('Token invalide ou expiré', 401);
        }
        
        ApiResponse::success([
            'valid' => true,
            'user_id' => $payload['user_id'],
            'expires_at' => date('c', $payload['exp'])
        ], 'Token valide');
    }
    
    private function refreshToken() {
        $user = Auth::requireAuth();
        
        try {
            // Récupérer les informations utilisateur actuelles
            $stmt = $this->db->prepare("SELECT id, nom, email, role FROM utilisateurs WHERE id = ? AND actif = 1");
            $stmt->execute([$user['user_id']]);
            $userData = $stmt->fetch();
            
            if (!$userData) {
                ApiResponse::error('Utilisateur non trouvé ou inactif', 404);
            }
            
            // Générer un nouveau token
            $newToken = Auth::generateToken($userData['id'], [
                'nom' => $userData['nom'],
                'email' => $userData['email'],
                'role' => $userData['role']
            ]);
            
            ApiResponse::success([
                'token' => $newToken,
                'user' => [
                    'id' => $userData['id'],
                    'nom' => $userData['nom'],
                    'email' => $userData['email'],
                    'role' => $userData['role']
                ]
            ], 'Token renouvelé');
            
        } catch (Exception $e) {
            Utils::logError('Erreur lors du renouvellement de token: ' . $e->getMessage());
            ApiResponse::error('Erreur lors du renouvellement du token', 500);
        }
    }
    
    private function logout() {
        // Dans un vrai système, on pourrait maintenir une liste de tokens révoqués
        // Pour cette implémentation simple, on se contente de confirmer la déconnexion
        
        $user = Auth::getCurrentUser();
        
        if ($user) {
            // Log de la déconnexion
            Utils::logError('Utilisateur déconnecté', ['user_id' => $user['user_id']]);
        }
        
        ApiResponse::success(null, 'Déconnexion réussie');
    }
}

// Point d'entrée de l'API
try {
    $api = new AuthAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API auth: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>