<?php
// api/achats.php - Endpoint pour la gestion des achats

require_once 'config.php';

class AchatsAPI {
    private $achat;
    private $user;
    
    public function __construct() {
        $this->achat = new Achat();
        $this->user = Auth::requireAuth();
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        $id = isset($pathParts[2]) ? (int)$pathParts[2] : null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getAchat($id);
                } else {
                    $this->getAchats();
                }
                break;
                
            case 'POST':
                $this->createAchat();
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateAchat($id);
                } else {
                    ApiResponse::error('ID requis pour la mise à jour', 400);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteAchat($id);
                } else {
                    ApiResponse::error('ID requis pour la suppression', 400);
                }
                break;
                
            default:
                ApiResponse::error('Méthode non supportée', 405);
        }
    }
    
    private function getAchats() {
        $where = [];
        $orderBy = 'date_achat DESC';
        
        // Filtres optionnels
        if (isset($_GET['site_id'])) {
            $where['site_id'] = (int)$_GET['site_id'];
        }
        
        if (isset($_GET['fournisseur_id'])) {
            $where['fournisseur_id'] = (int)$_GET['fournisseur_id'];
        }
        
        if (isset($_GET['date_debut']) && isset($_GET['date_fin'])) {
            // Pour les plages de dates, utiliser une requête personnalisée
            $achats = $this->getAchatsByDateRange($_GET['date_debut'], $_GET['date_fin'], $where);
        } else {
            $achats = $this->achat->findAll($where, $orderBy, $_GET['limit'] ?? 100);
        }
        
        // Enrichir avec les informations liées
        foreach ($achats as &$achat) {
            $achat['site'] = $this->getSiteInfo($achat['site_id']);
            $achat['fournisseur'] = $this->getFournisseurInfo($achat['fournisseur_id']);
            $achat['produit'] = $this->getProduitInfo($achat['produit_id']);
        }
        
        ApiResponse::success($achats, 'Achats récupérés');
    }
    
    private function getAchatsByDateRange($dateDebut, $dateFin, $additionalWhere = []) {
        $sql = "SELECT * FROM achats WHERE date_achat BETWEEN ? AND ?";
        $params = [$dateDebut, $dateFin];
        
        foreach ($additionalWhere as $field => $value) {
            $sql .= " AND $field = ?";
            $params[] = $value;
        }
        
        $sql .= " ORDER BY date_achat DESC";
        
        $stmt = $this->achat->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }
    
    private function getAchat($id) {
        $achat = $this->achat->find($id);
        
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        // Enrichir avec les informations liées
        $achat['site'] = $this->getSiteInfo($achat['site_id']);
        $achat['fournisseur'] = $this->getFournisseurInfo($achat['fournisseur_id']);
        $achat['produit'] = $this->getProduitInfo($achat['produit_id']);
        
        ApiResponse::success($achat, 'Achat récupéré');
    }
    
    private function createAchat() {
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->achat->validate($data);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        // Nettoyer les données
        $data = Utils::sanitizeInput($data);
        $data['utilisateur_id'] = $this->user['user_id'];
        
        // Calculer les totaux
        $achatObj = new Achat();
        foreach ($data as $key => $value) {
            $achatObj->$key = $value;
        }
        $achatObj->calculateTotals();
        
        $data['montant_total'] = $achatObj->montant_total;
        $data['cout_total'] = $achatObj->cout_total;
        
        try {
            $this->achat->beginTransaction();
            
            // Vérifier le budget disponible
            $this->checkBudgetAvailable($data['site_id'], $data['cout_total']);
            
            // Créer l'achat
            $id = $this->achat->create($data);
            
            // Mettre à jour le stock
            $achatCreated = $this->achat->find($id);
            $achatObj = new Achat();
            foreach ($achatCreated as $key => $value) {
                $achatObj->$key = $value;
            }
            $achatObj->updateStock();
            
            // Mettre à jour le budget
            $this->updateBudget($data['site_id'], $data['cout_total'], 'subtract');
            
            // Créer la transaction financière
            $this->createTransaction($data['site_id'], 'sortie', 'achat', $data['cout_total'], 
                "Achat - " . $this->getProduitName($data['produit_id']), $id);
            
            $this->achat->commit();
            
            ApiResponse::success(['id' => $id], 'Achat créé avec succès', 201);
            
        } catch (Exception $e) {
            $this->achat->rollback();
            Utils::logError('Erreur création achat: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la création de l\'achat: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateAchat($id) {
        $achat = $this->achat->find($id);
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        $data = Utils::parseJsonInput();
        
        // Validation
        $errors = $this->achat->validate($data, true);
        if (!empty($errors)) {
            ApiResponse::validation($errors);
        }
        
        $data = Utils::sanitizeInput($data);
        
        try {
            $success = $this->achat->update($id, $data);
            
            if ($success) {
                ApiResponse::success(null, 'Achat mis à jour avec succès');
            } else {
                ApiResponse::error('Échec de la mise à jour', 500);
            }
            
        } catch (Exception $e) {
            Utils::logError('Erreur mise à jour achat: ' . $e->getMessage(), $data);
            ApiResponse::error('Erreur lors de la mise à jour de l\'achat', 500);
        }
    }
    
    private function deleteAchat($id) {
        $achat = $this->achat->find($id);
        if (!$achat) {
            ApiResponse::error('Achat non trouvé', 404);
        }
        
        try {
            $this->achat->beginTransaction();
            
            // Logique de suppression complexe (inverser les mouvements de stock, etc.)
            // Pour l'instant, simple suppression
            $success = $this->achat->delete($id);
            
            if ($success) {
                $this->achat->commit();
                ApiResponse::success(null, 'Achat supprimé avec succès');
            } else {
                $this->achat->rollback();
                ApiResponse::error('Échec de la suppression', 500);
            }
            
        } catch (Exception $e) {
            $this->achat->rollback();
            Utils::logError('Erreur suppression achat: ' . $e->getMessage());
            ApiResponse::error('Erreur lors de la suppression de l\'achat', 500);
        }
    }
    
    // Méthodes utilitaires
    private function checkBudgetAvailable($siteId, $montant) {
        $budget = new Budget();
        $budgets = $budget->findAll(['site_id' => $siteId, 'statut' => 'actif']);
        
        $budgetTotal = array_sum(array_column($budgets, 'montant_actuel'));
        
        if ($budgetTotal < $montant) {
            throw new Exception('Budget insuffisant pour cet achat');
        }
    }
    
    private function updateBudget($siteId, $montant, $operation) {
        $budget = new Budget();
        $budgets = $budget->findAll(['site_id' => $siteId, 'statut' => 'actif'], 'date_creation ASC');
        
        $montantRestant = $montant;
        
        foreach ($budgets as $budgetData) {
            if ($montantRestant <= 0) break;
            
            $budgetObj = new Budget();
            foreach ($budgetData as $key => $value) {
                $budgetObj->$key = $value;
            }
            
            if ($operation === 'subtract') {
                $aDeduire = min($montantRestant, $budgetData['montant_actuel']);
                $budgetObj->updateAmount($aDeduire, 'subtract');
                $montantRestant -= $aDeduire;
            }
        }
    }
    
    private function createTransaction($siteId, $type, $categorie, $montant, $description, $referenceId) {
        $stmt = $this->achat->db->prepare("
            INSERT INTO transactions (site_id, type_transaction, categorie, montant, description, reference_id, date_transaction, utilisateur_id)
            VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?)
        ");
        
        return $stmt->execute([
            $siteId, $type, $categorie, $montant, $description, $referenceId, $this->user['user_id']
        ]);
    }
    
    private function getSiteInfo($siteId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, type_site FROM sites WHERE id = ?");
        $stmt->execute([$siteId]);
        return $stmt->fetch();
    }
    
    private function getFournisseurInfo($fournisseurId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, telephone FROM fournisseurs WHERE id = ?");
        $stmt->execute([$fournisseurId]);
        return $stmt->fetch();
    }
    
    private function getProduitInfo($produitId) {
        $stmt = $this->achat->db->prepare("SELECT id, nom, unite_mesure, categorie FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        return $stmt->fetch();
    }
    
    private function getProduitName($produitId) {
        $stmt = $this->achat->db->prepare("SELECT nom FROM produits WHERE id = ?");
        $stmt->execute([$produitId]);
        $result = $stmt->fetch();
        return $result ? $result['nom'] : 'Produit inconnu';
    }
}

try {
    $api = new AchatsAPI();
    $api->handleRequest();
} catch (Exception $e) {
    Utils::logError('Erreur API achats: ' . $e->getMessage());
    ApiResponse::error('Erreur interne du serveur', 500);
}

?>